#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import logging
logger = logging.getLogger('PyAMS (content)')

from pyramid.decorator import reify
from zope.interface import implementer

from pyams_cache.beaker import get_cache
from pyams_content.features.renderer.interfaces import HIDDEN_RENDERER_NAME
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_portal.interfaces import PREVIEW_MODE
from pyams_portal.portlet import PORTLETS_CACHE_KEY, PORTLETS_CACHE_REGION, PORTLETS_CACHE_NAME
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.traversing import get_parent
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.features.footer.interfaces import IFooterTarget, IFooterRenderer, \
    IFooterSettings, IFooterRendererSettings
from pyams_default_theme.features.footer.interfaces import ISimpleFooterRendererSettings
from pyams_skin.layer import IPyAMSLayer
from pyams_default_theme.features.footer import SimpleFooterRendererSettings
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


#
# Simple footer renderer
#

@implementer(IFooterRenderer)
class BaseFooterRenderer(BaseContentRenderer):
    """Base footer renderer"""

    name = None
    settings_key = None

    @reify
    def settings_target(self):
        context = self.request.annotations.get(DISPLAY_CONTEXT)
        if context is None:
            context = self.context
        return get_parent(context, IFooterTarget)

    @reify
    def settings(self):
        if self.settings_interface is None:
            return None
        target = self.settings_target
        settings = IFooterSettings(target)
        while settings.inherit:
            settings = IFooterSettings(settings.parent)
        return settings.settings

    @reify
    def cache_key(self):
        return PORTLETS_CACHE_KEY.format(portlet=self.name,
                                         hostname=self.request.host,
                                         context=ICacheKeyValue(self.settings_target),
                                         lang=self.request.locale_name)

    def render(self, template_name=''):
        preview_mode = self.request.annotations.get(PREVIEW_MODE, False)
        if preview_mode:
            return super().render(template_name)
        portlets_cache = get_cache(PORTLETS_CACHE_REGION, PORTLETS_CACHE_NAME)
        cache_key = self.cache_key
        if self.context is not self.request.context:  # display shared content
            cache_key = '{}::shared'.format(cache_key)
        if template_name:
            cache_key = '{}::{}'.format(cache_key, template_name)
        try:
            result = portlets_cache.get_value(cache_key)
            logger.debug("Retrieving footer content from cache key {}".format(cache_key))
        except KeyError:
            result = super().render(template_name)
            portlets_cache.set_value(cache_key, result)
            logger.debug("Storing footer content to cache key {}".format(cache_key))
        return result


#
# Hidden footer renderer
#

@adapter_config(name=HIDDEN_RENDERER_NAME, context=(IFooterTarget, IPyAMSLayer), provides=IFooterRenderer)
class HiddenFooterRenderer(BaseFooterRenderer):
    """Hidden footer renderer"""

    name = HIDDEN_RENDERER_NAME
    label = _("Hidden footer")
    weight = -999

    def render(self, template_name=''):
        return ''


#
# Simple footer renderer
#

SIMPLE_FOOTER_RENDERER_NAME = 'PyAMS simple footer'


@adapter_config(name=SIMPLE_FOOTER_RENDERER_NAME, context=(IFooterTarget, IPyAMSLayer),
                provides=IFooterRenderer)
@adapter_config(name=SIMPLE_FOOTER_RENDERER_NAME, context=(IFooterSettings, IPyAMSLayer),
                provides=IFooterRenderer)
@template_config(template='templates/simple-footer.pt', layer=IPyAMSLayer)
class SimpleFooterRenderer(BaseFooterRenderer):
    """Simple footer renderer"""

    name = SIMPLE_FOOTER_RENDERER_NAME
    label = _("PyAMS simple footer with links")
    weight = 1

    settings_key = 'PyAMS::simple'
    settings_interface = ISimpleFooterRendererSettings

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)


@adapter_config(context=SimpleFooterRenderer, provides=IFooterRendererSettings)
def simple_footer_renderer_settings_factory(context):
    """Simple footer renderer settings factory"""
    return SimpleFooterRendererSettings()
