#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration.interfaces import IIllustration
from pyams_content.component.verbatim.interfaces import IVerbatimParagraph
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.component.verbatim.interfaces import \
    IBaseVerbatimParagraphRendererSettings, ILateralVerbatimParagraphRendererSettings, \
    IVerbatimParagraphRendererSettings
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config

from pyams_default_theme import _


class BaseVerbatimParagraphRendererSettings(Persistent, Location):
    """Base verbatim paragraph renderer settings"""
    zoom_on_click = FieldProperty(IBaseVerbatimParagraphRendererSettings['zoom_on_click'])


#
# Verbatim paragraph renderer settings
#

VERBATIM_PARAGRAPH_RENDERER_SETTINGS_KEY = 'pyams_content.verbatim.renderer'


@factory_config(IVerbatimParagraphRendererSettings)
class VerbatimParagraphRendererSettings(BaseVerbatimParagraphRendererSettings):
    """Verbatim paragraph renderer settings"""

    xs_horizontal_padding = FieldProperty(
        IVerbatimParagraphRendererSettings['xs_horizontal_padding'])
    sm_horizontal_padding = FieldProperty(
        IVerbatimParagraphRendererSettings['sm_horizontal_padding'])
    md_horizontal_padding = FieldProperty(
        IVerbatimParagraphRendererSettings['md_horizontal_padding'])
    lg_horizontal_padding = FieldProperty(
        IVerbatimParagraphRendererSettings['lg_horizontal_padding'])

    def has_padding(self):
        return self.lg_horizontal_padding or \
               self.md_horizontal_padding or \
               self.sm_horizontal_padding or \
               self.xs_horizontal_padding

    def get_padding(self):
        result = []
        for device in ('xs', 'sm', 'md', 'lg'):
            padding = getattr(self, '{}_horizontal_padding'.format(device))
            if padding is not None:
                result.append('col-{0}-{1} col-{0}-offset-{2}'.format(device,
                                                                      12 - (padding * 2),
                                                                      padding))
        return ' '.join(result)


@adapter_config(context=IVerbatimParagraph,
                provides=IVerbatimParagraphRendererSettings)
def verbatim_paragraph_renderer_settings_factory(context):
    """Frame text paragraph renderer settings factory"""
    return get_annotation_adapter(context, VERBATIM_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  IVerbatimParagraphRendererSettings)


#
# Verbatim paragraph lateral renderer settings
#

LATERAL_VERBATIM_PARAGRAPH_RENDERER_SETTINGS_KEY = 'pyams_content.verbatim.renderer:lateral'


@factory_config(ILateralVerbatimParagraphRendererSettings)
class LateralVerbatimParagraphRendererSettings(BaseVerbatimParagraphRendererSettings):
    """Verbatim paragraph lateral renderer settings"""

    relative_width = FieldProperty(ILateralVerbatimParagraphRendererSettings['relative_width'])


@adapter_config(context=IVerbatimParagraph,
                provides=ILateralVerbatimParagraphRendererSettings)
def lateral_verbatim_paragraph_renderer_settings_factory(context):
    """Frame text paragraph lateral renderer settings factory"""
    return get_annotation_adapter(context, LATERAL_VERBATIM_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  ILateralVerbatimParagraphRendererSettings)


#
# Verbatim paragraph default renderer
#

class BaseVerbatimParagraphRenderer(BaseContentRenderer):
    """Base frame paragraph renderer"""

    context_attrs = ('author',)
    i18n_context_attrs = ('title', 'quote', 'charge')

    def can_display_illustration(self):
        frame = IVerbatimParagraph(self.context)
        illustration = IIllustration(frame, None)
        return (illustration is not None) and bool(II18n(illustration).query_attribute('data'))


@adapter_config(name='default',
                context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/verbatim-default.pt', layer=IPyAMSLayer)
class DefaultVerbatimParagraphRenderer(BaseVerbatimParagraphRenderer):
    """Verbatim paragraph default renderer"""

    label = _("Default verbatim renderer")
    weight = 1

    settings_interface = IVerbatimParagraphRendererSettings


@adapter_config(name='left',
                context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/verbatim-left.pt', layer=IPyAMSLayer)
class LeftVerbatimParagraphRenderer(BaseVerbatimParagraphRenderer):
    """Verbatim paragraph renderer displayed on the left"""

    label = _("Small frame on the left")
    weight = 2

    settings_interface = ILateralVerbatimParagraphRendererSettings


@adapter_config(name='right',
                context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/verbatim-right.pt', layer=IPyAMSLayer)
class RightVerbatimParagraphRenderer(BaseVerbatimParagraphRenderer):
    """Verbatim paragraph renderer displayed on the right"""

    label = _("Small frame on the right")
    weight = 3

    settings_interface = ILateralVerbatimParagraphRendererSettings
