#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import Interface

from pyams_content.component.paragraph.interfaces import IParagraphContainer
from pyams_content.component.paragraph.portlet.interfaces import IParagraphContainerPortletSettings, \
    IParagraphNavigationPortletSettings
from pyams_content.features.renderer.interfaces import ISharedContentRenderer
from pyams_content.shared.site.interfaces import ISiteContainer
from pyams_default_theme.component.paragraph.interfaces import IParagraphContainerPortletRenderer
from pyams_default_theme.interfaces import IContentSummaryInfo
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.portlet import PortletRenderer
from pyams_sequence.interfaces import IInternalReference, ISequentialIdInfo
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


@adapter_config(context=(IPortalContext, IPyAMSLayer, Interface,
                         IParagraphContainerPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/content.pt', layer=IPyAMSLayer)
class ParagraphContainerPortletRenderer(PortletRenderer):
    """Paragraph container default portlet renderer"""

    label = _("Default paragraphs renderer")

    renderers = ()
    custom_renderer = None

    template_name = ''

    def update(self):
        super().update()
        settings = self.settings
        if settings.reference:
            source = settings.target
        else:
            source = self.context
        registry = self.request.registry
        renderer = self.custom_renderer = registry.queryMultiAdapter(
            (source, self.request, self.view), IParagraphContainerPortletRenderer,
            name=self.request.view_name)
        if renderer is not None:
            renderer.update()
            self.renderers = [renderer]
        else:
            container = IParagraphContainer(source, None)
            if container is not None:
                paragraphs = container.get_visible_paragraphs(
                    names=settings.paragraphs,
                    anchors_only=settings.anchors_only,
                    exclude_anchors=settings.exclude_anchors,
                    factories=settings.factories,
                    limit=settings.limit)
                renderers = [
                    paragraph.get_renderer(self.request)
                    for paragraph in paragraphs
                ]
            else:
                renderers = [
                    adapter
                    for name, adapter in sorted(registry.getAdapters((source, self.request),
                                                                     ISharedContentRenderer),
                                                key=lambda x: x[1].weight)
                ]
            self.renderers = list(filter(lambda x: x is not None, renderers))
            [renderer.update() for renderer in self.renderers]

    @property
    def use_portlets_cache(self):
        use_cache = super().use_portlets_cache
        if use_cache and (self.custom_renderer is not None):
            return self.custom_renderer.use_portlets_cache

    def get_cache_key(self):
        key = super().get_cache_key()
        if self.custom_renderer is not None:
            key = '{}::{}'.format(key, self.request.view_name)
        return key

    def get_navigation_links(self):

        def test_item(nav_item):
            item_sequence = ISequentialIdInfo(nav_item, None)
            if (item_sequence is not None) and (item_sequence.oid == context_sequence.oid):
                return True
            if IInternalReference.providedBy(nav_item) and \
                    (nav_item.reference == context_sequence.hex_oid):
                return True

        prev_nav, next_nav = None, None
        context_sequence = ISequentialIdInfo(self.context, None)
        if context_sequence is not None:
            display_context = self.request.display_context
            if ISiteContainer.providedBy(display_context):
                registry = self.request.registry
                previous_item = None
                items = display_context.get_visible_items(self.request)
                for item in items:
                    if test_item(item):
                        prev_nav = registry.queryMultiAdapter((previous_item, self.request),
                                                              IContentSummaryInfo)
                        break
                    previous_item = item
                try:
                    next_item = next(items)
                except (StopIteration, RuntimeError):
                    pass
                else:
                    next_nav = registry.queryMultiAdapter((next_item, self.request),
                                                          IContentSummaryInfo)
        return prev_nav, next_nav


@adapter_config(context=(IPortalContext, IPyAMSLayer, Interface,
                         IParagraphNavigationPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/navigation.pt', layer=IPyAMSLayer)
class ParagraphNavigationPortletRenderer(PortletRenderer):
    """Paragraphs navigation default portlet renderer"""

    label = _("Default paragraphs navigation")

    paragraphs = ()

    def update(self):
        super().update()
        settings = self.settings
        container = IParagraphContainer(self.context, None)
        if container is not None:
            self.paragraphs = container.get_visible_paragraphs(
                names=settings.paragraphs,
                anchors_only=settings.anchors_only,
                factories=settings.factories)
