#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from fanstatic import ConfigurationError
from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration import IIllustration
from pyams_content.component.links import InternalReferenceMixin
from pyams_content.component.paragraph.interfaces.html import IHTMLParagraph, IRawParagraph
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme import library
from pyams_default_theme.component.paragraph.interfaces.html import \
    HTML_PARAGRAPH_DEFAULT_RENDERER_SETTINGS_KEY, IHTMLParagraphDefaultRendererSettings, \
    IHTMLParagraphWithLogosRendererSettings
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_default_theme.shared.logo import DISABLED_LINK, INTERNAL_FIRST
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.fanstatic import ExternalResource
from pyams_utils.interfaces.text import IHTMLRenderer
from pyams_utils.pygments import IPygmentsCodeConfiguration, render_source
from pyams_utils.url import canonical_url, relative_url


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


#
# Raw paragraph default renderer
#

@adapter_config(name='default', context=(IRawParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/raw-default.pt', layer=IPyAMSLayer)
class RawParagraphDefaultRenderer(BaseContentRenderer):
    """Raw paragraph default renderer"""

    label = _("Default raw HTML renderer")
    weight = 1

    i18n_context_attrs = ('title', 'body')


#
# Raw paragraph source code renderer
#

RAW_PARAGRAPH_SOURCE_CODE_RENDERER_SETTINGS_KEY = 'pyams_content.raw.renderer:source-code'


@adapter_config(context=IRawParagraph, provides=IPygmentsCodeConfiguration)
def raw_paragraph_source_code_renderer_settings_factory(context):
    """Raw paragraph source code renderer settings factory"""
    return get_annotation_adapter(context, RAW_PARAGRAPH_SOURCE_CODE_RENDERER_SETTINGS_KEY,
                                  IPygmentsCodeConfiguration)


@adapter_config(name='source-code', context=(IRawParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/raw-source-code.pt', layer=IPyAMSLayer)
class RawParagraphSourceCodeRenderer(BaseContentRenderer):
    """Raw paragraph source code renderer"""

    label = _("Formatted source code renderer")
    weight = 10

    settings_interface = IPygmentsCodeConfiguration

    i18n_context_attrs = ('title', 'body')

    @property
    def resources(self):
        path = 'get-pygments-style.css?style={}'.format(self.settings.style)
        try:
            yield ExternalResource(library, path, resource_type='css')
        except ConfigurationError:
            yield library.known_resources[path]

    @property
    def source(self):
        code = self.body
        if code:
            return render_source(code, self.settings)
        return ''


@adapter_config(name='rest', context=(IRawParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/raw-rest.pt', layer=IPyAMSLayer)
class RawParagraphRestRenderer(BaseContentRenderer):
    """Raw paragraph ReStructuredText code renderer"""

    label = _("ReStructuredText renderer")
    weight = 20

    i18n_context_attrs = ('title', 'body')

    @property
    def source(self):
        code = self.body
        if code:
            renderer = self.request.registry.queryMultiAdapter((code, self.request), IHTMLRenderer,
                                                               name='rest')
            if renderer is not None:
                return renderer.render()
        return ''


@adapter_config(name='markdown', context=(IRawParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/raw-markdown.pt', layer=IPyAMSLayer)
class RawParagraphMarkdownRenderer(BaseContentRenderer):
    """Raw paragraph markdown code renderer"""

    label = _("Markdown renderer")
    weight = 30

    i18n_context_attrs = ('title', 'body')

    @property
    def source(self):
        code = self.body
        if code:
            renderer = self.request.registry.queryMultiAdapter((code, self.request), IHTMLRenderer,
                                                               name='markdown')
            if renderer is not None:
                return renderer.render()
        return ''


#
# HTML paragraph default renderer
#

@factory_config(provided=IHTMLParagraphDefaultRendererSettings)
class HTMLParagraphDefaultRendererSettings(Persistent, Location):
    """HTML paragraph default renderer settings"""

    xs_horizontal_padding = FieldProperty(
        IHTMLParagraphDefaultRendererSettings['xs_horizontal_padding'])
    sm_horizontal_padding = FieldProperty(
        IHTMLParagraphDefaultRendererSettings['sm_horizontal_padding'])
    md_horizontal_padding = FieldProperty(
        IHTMLParagraphDefaultRendererSettings['md_horizontal_padding'])
    lg_horizontal_padding = FieldProperty(
        IHTMLParagraphDefaultRendererSettings['lg_horizontal_padding'])

    def has_padding(self):
        return self.lg_horizontal_padding or \
               self.md_horizontal_padding or \
               self.sm_horizontal_padding or \
               self.xs_horizontal_padding

    def get_padding(self):
        result = []
        for device in ('xs', 'sm', 'md', 'lg'):
            padding = getattr(self, '{}_horizontal_padding'.format(device))
            if padding is not None:
                result.append('col-{0}-{1} col-{0}-offset-{2}'.format(device,
                                                                      12 - (padding * 2),
                                                                      padding))
        return ' '.join(result)


@adapter_config(context=IHTMLParagraph,
                provides=IHTMLParagraphDefaultRendererSettings)
def html_paragraph_default_renderer_settings_factory(context):
    """HTML paragraph default renderer settings factory"""
    return get_annotation_adapter(context, HTML_PARAGRAPH_DEFAULT_RENDERER_SETTINGS_KEY,
                                  IHTMLParagraphDefaultRendererSettings)


@adapter_config(name='default', context=(IHTMLParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/html-default.pt', layer=IPyAMSLayer)
class HTMLParagraphDefaultRenderer(BaseContentRenderer):
    """HTML paragraph default renderer"""

    label = _("Default rich text renderer")
    weight = 10

    settings_interface = IHTMLParagraphDefaultRendererSettings

    i18n_context_attrs = ('title', 'body',)

    illustration = None
    illustration_renderer = None

    def update(self):
        super(HTMLParagraphDefaultRenderer, self).update()
        self.illustration = IIllustration(self.context)
        if self.illustration.data:
            renderer = self.illustration_renderer = self.illustration.get_renderer(self.request)
            if renderer is not None:
                renderer.update()


#
# HTML paragraph renderer with logos
#

HTML_PARAGRAPH_WITH_LOGOS_RENDERER_SETTINGS_KEY = 'pyams_content.html.renderer:text-with-logos'


@factory_config(IHTMLParagraphWithLogosRendererSettings)
class HTMLParagraphWithLogosRendererSettings(Persistent, Location, InternalReferenceMixin):
    """HTML paragraph with logos renderer settings"""

    reference = FieldProperty(IHTMLParagraphWithLogosRendererSettings['reference'])
    position = FieldProperty(IHTMLParagraphWithLogosRendererSettings['position'])
    target_priority = FieldProperty(IHTMLParagraphWithLogosRendererSettings['target_priority'])
    force_canonical_url = FieldProperty(IHTMLParagraphWithLogosRendererSettings['force_canonical_url'])


@adapter_config(context=IHTMLParagraph, provides=IHTMLParagraphWithLogosRendererSettings)
def html_paragraph_with_logos_renderer_settings_factory(context):
    """HTML paragraph with logos renderer settings factory"""
    return get_annotation_adapter(context, HTML_PARAGRAPH_WITH_LOGOS_RENDERER_SETTINGS_KEY,
                                  IHTMLParagraphWithLogosRendererSettings)


@adapter_config(name='text-with-logos',
                context=(IHTMLParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/html-logos.pt', layer=IPyAMSLayer)
class HTMLParagraphWithLogosRenderer(HTMLParagraphDefaultRenderer):
    """HTML paragraph renderer with logos"""

    label = _("Rich text renderer with logos")
    weight = 20

    settings_interface = IHTMLParagraphWithLogosRendererSettings

    def get_internal_url(self, logo):
        if logo is not None:
            if self.settings.force_canonical_url:
                url_getter = canonical_url
            else:
                url_getter = relative_url
            return url_getter(logo, request=self.request)
        return None

    @staticmethod
    def get_external_url(logo):
        return logo.url

    def get_logo_url(self):
        priority = self.settings.target_priority
        if priority == DISABLED_LINK:
            return None
        logo = self.settings.target
        if logo is None:
            return None
        order = [self.get_external_url, self.get_internal_url]
        if priority == INTERNAL_FIRST:
            order = reversed(order)
        for getter in order:
            result = getter(logo)
            if result is not None:
                return result
        return None
