#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.interface import implementer
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.association.interfaces import IAssociationContainer
from pyams_content.component.illustration.interfaces import IIllustration
from pyams_content.component.paragraph.interfaces.frame import IFrameParagraph
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_content.reference.pictograms.interfaces import IPictogramTable
from pyams_default_theme.component.paragraph.interfaces.frame import IDefaultFrameParagraphRendererSettings, \
    IFrameParagraphRendererSettings, ILateralFrameParagraphRendererSettings, ILeftFrameParagraphRendererSettings, \
    IRightFrameParagraphRendererSettings
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility
from pyams_utils.zodb import volatile_property

from pyams_default_theme import _


#
# Framed text paragraph default renderer settings
#

FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY = 'pyams_content.frame.renderer:default'
LEFT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY = 'pyams_content.frame.renderer:left'
RIGHT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY = 'pyams_content.frame.renderer:right'


@implementer(IFrameParagraphRendererSettings)
class BaseFrameParagraphRendererSettings(Persistent, Location):
    """Base frame text paragraph renderer settings"""

    display_associations = FieldProperty(IFrameParagraphRendererSettings['display_associations'])
    _pictogram_name = FieldProperty(IDefaultFrameParagraphRendererSettings['pictogram_name'])

    def can_display_associations(self):
        if not self.display_associations:
            return False
        frame = IFrameParagraph(self.__parent__)
        associations = IAssociationContainer(frame, None)
        return (associations is not None) and (len(associations.get_visible_items()) > 0)

    @property
    def pictogram_name(self):
        return self._pictogram_name

    @pictogram_name.setter
    def pictogram_name(self, value):
        if value != self._pictogram_name:
            self._pictogram_name = value
            del self.pictogram

    @volatile_property
    def pictogram(self):
        table = query_utility(IPictogramTable)
        if table is not None:
            return table.get(self._pictogram_name)


@factory_config(IDefaultFrameParagraphRendererSettings)
class DefaultFrameParagraphRendererSettings(BaseFrameParagraphRendererSettings):
    """Framed text paragraph lateral renderer settings"""


@implementer(ILateralFrameParagraphRendererSettings)
class LateralFrameParagraphRendererSettings(BaseFrameParagraphRendererSettings):
    """Framed text paragraph lateral renderer settings"""

    relative_width = FieldProperty(ILateralFrameParagraphRendererSettings['relative_width'])


@factory_config(ILeftFrameParagraphRendererSettings)
class LeftFrameParagraphRendererSettings(LateralFrameParagraphRendererSettings):
    """Left framed text paragraph lateral renderer settings"""


@factory_config(IRightFrameParagraphRendererSettings)
class RightFrameParagraphRendererSettings(LateralFrameParagraphRendererSettings):
    """Right framed text paragraph lateral renderer settings"""


#
# Framed text paragraph renderers settings
#

@adapter_config(context=IFrameParagraph, provides=IDefaultFrameParagraphRendererSettings)
def default_frame_paragraph_renderer_settings_factory(context):
    """Frame paragraph default renderer settings factory"""
    return get_annotation_adapter(context, FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  IDefaultFrameParagraphRendererSettings)


@adapter_config(context=IFrameParagraph, provides=ILeftFrameParagraphRendererSettings)
def left_lateral_frame_paragraph_renderer_settings_factory(context):
    """Frame text paragraph lateral renderer settings factory"""
    return get_annotation_adapter(context, LEFT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  ILeftFrameParagraphRendererSettings)


@adapter_config(context=IFrameParagraph, provides=IRightFrameParagraphRendererSettings)
def right_lateral_frame_paragraph_renderer_settings_factory(context):
    """Frame text paragraph lateral renderer settings factory"""
    return get_annotation_adapter(context, RIGHT_FRAME_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  IRightFrameParagraphRendererSettings)


#
# Framed text paragraph renderers settings
#

class BaseFrameParagraphRenderer(BaseContentRenderer):
    """Base frame paragraph renderer"""

    i18n_context_attrs = ('title', 'body')

    illustration = None
    illustration_renderer = None

    def update(self):
        super(BaseFrameParagraphRenderer, self).update()
        self.illustration = IIllustration(self.context)
        if self.illustration.data:
            renderer = self.illustration_renderer = self.illustration.get_renderer(self.request)
            if renderer is not None:
                renderer.update()


@adapter_config(name='default', context=(IFrameParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/frame-default.pt', layer=IPyAMSLayer)
class DefaultFrameParagraphRenderer(BaseFrameParagraphRenderer):
    """Framed text paragraph default renderer"""

    label = _("Default frame renderer")
    weight = 1

    settings_interface = IDefaultFrameParagraphRendererSettings


@adapter_config(name='left', context=(IFrameParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/frame-left.pt', layer=IPyAMSLayer)
class LeftFrameParagraphRenderer(BaseFrameParagraphRenderer):
    """Framed text paragraph renderer displayed on the left"""

    label = _("Small frame on the left")
    weight = 2

    settings_interface = ILeftFrameParagraphRendererSettings


@adapter_config(name='right', context=(IFrameParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/frame-right.pt', layer=IPyAMSLayer)
class RightFrameParagraphRenderer(BaseFrameParagraphRenderer):
    """Framed text paragraph renderer displayed on the right"""

    label = _("Small frame on the right")
    weight = 3

    settings_interface = IRightFrameParagraphRendererSettings
