#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
import json

from pyramid.encode import url_quote


__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph.interfaces.contact import IContactParagraph, have_gis
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.component.paragraph.interfaces.contact import IContactParagraphDefaultRendererSettings
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_gis.interfaces.utility import IMapManager
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config

from pyams_default_theme import _


#
# Contact paragraph default renderer settings
#
from pyams_utils.registry import query_utility


CONTACT_DEFAULT_RENDERER_SETTINGS_KEY = 'pyams_content.contact.renderer:default'


@factory_config(IContactParagraphDefaultRendererSettings)
class ContactParagraphDefaultRendererSettings(Persistent, Location):
    """Contact paragraph default renderer settings"""

    display_photo = FieldProperty(IContactParagraphDefaultRendererSettings['display_photo'])
    photo_position = FieldProperty(IContactParagraphDefaultRendererSettings['photo_position'])
    display_map = FieldProperty(IContactParagraphDefaultRendererSettings['display_map'])
    display_coordinates = FieldProperty(
        IContactParagraphDefaultRendererSettings['display_coordinates'])
    display_marker = FieldProperty(IContactParagraphDefaultRendererSettings['display_marker'])
    map_position = FieldProperty(IContactParagraphDefaultRendererSettings['map_position'])
    button_label = FieldProperty(IContactParagraphDefaultRendererSettings['button_label'])

    @property
    def can_display_photo(self):
        contact = IContactParagraph(self.__parent__)
        if not contact.photo:
            return False
        return self.display_photo

    @property
    def can_display_map(self):
        if not have_gis:
            return False
        contact = IContactParagraph(self.__parent__)
        if not contact.gps_location:
            return False
        return self.display_map


@adapter_config(context=IContactParagraph, provides=IContactParagraphDefaultRendererSettings)
def contact_paragraph_default_renderer_settings_factory(context):
    """Contact paragraph default renderer settings factory"""
    return get_annotation_adapter(context, CONTACT_DEFAULT_RENDERER_SETTINGS_KEY,
                                  IContactParagraphDefaultRendererSettings)


#
# Contact paragraph default renderer
#

@adapter_config(name='default', context=(IContactParagraph, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/contact-default.pt', layer=IPyAMSLayer)
class ContactParagraphDefaultRenderer(BaseContentRenderer):
    """Context paragraph default renderer"""

    label = _("Default contact renderer")

    settings_interface = IContactParagraphDefaultRendererSettings

    context_attrs = ('name', 'photo', 'company', 'address', 'phone_number', 'contact_email')
    i18n_context_attrs = ('title', 'charge',)

    @property
    def gps_location(self):
        if not have_gis:
            return None
        return self.context.gps_location

    def get_map_configuration(self):
        manager = query_utility(IMapManager)
        if manager is not None:
            configuration = IMapConfiguration(manager).get_configuration()
            if self.context.gps_location and self.settings.display_marker:
                coordinates = self.context.gps_location.wgs_coordinates
                configuration['marker'] = {
                    'lat': float(coordinates[1]),
                    'lon': float(coordinates[0])
                }
            return json.dumps(configuration)

    @property
    def button_label(self):
        label = II18n(self.settings).query_attribute('button_label', request=self.request)
        if not label:
            label = self.request.localizer.translate(_("contact-button-label", default="Contact"))
        return label

    @property
    def contact_url(self):
        if self.name:
            return 'mailto:{}'.format(url_quote('{} <{}>'.format(self.name, self.contact_email)))
        else:
            return 'mailto:{}'.format(self.contact_email)

    @property
    def contact_subject(self):
        return url_quote(self.title)
