#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.interface import Interface, implementer
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration import IBasicIllustrationTarget, ILinkIllustration, \
    ILinkIllustrationTarget
from pyams_content.component.illustration.interfaces import IIllustration
from pyams_content.component.links import IInternalLink
from pyams_content.features.renderer.interfaces import HIDDEN_RENDERER_NAME, IContentRenderer
from pyams_content.interfaces import IBaseContent
from pyams_content.root import ISiteRoot
from pyams_content.shared.common import ISharedContent
from pyams_default_theme import lightbox
from pyams_default_theme.component.illustration.interfaces import IIllustrationRenderer, \
    IIllustrationWithZoomSettings, ILLUSTRATION_AFTER_BODY, ILLUSTRATION_BEFORE_BODY
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_default_theme.interfaces import IContentBannerIllustration, IContentHeaderIllustration, \
    IContentNavigationIllustration
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.interfaces.tales import ITALESExtension


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


#
# Illustrations adapters
#

@adapter_config(context=(IInternalLink, IPyAMSLayer), provides=IContentNavigationIllustration)
@adapter_config(context=(IBasicIllustrationTarget, IPyAMSLayer),
                provides=IContentNavigationIllustration)
def base_content_navigation_illustration_factory(context, request):
    """Default content navigation illustration adapter"""
    illustration = ILinkIllustration(context, None)
    if not (illustration and illustration.has_data()):
        illustration = IIllustration(context, None)
        if IIllustration.providedBy(illustration) and \
                (illustration.renderer == HIDDEN_RENDERER_NAME):
            illustration = None
    if illustration and illustration.has_data():
        return illustration
    if IInternalLink.providedBy(context):
        target = context.get_target()
        if target is not None:
            illustration = request.registry.queryMultiAdapter((target, request),
                                                              IContentNavigationIllustration)
            if illustration and illustration.has_data():
                return illustration


@adapter_config(context=(ISiteRoot, IPyAMSLayer), provides=IContentBannerIllustration)
@adapter_config(context=(IBaseContent, IPyAMSLayer), provides=IContentBannerIllustration)
def base_content_banner_illustration_factory(context, request):
    """Base content banner illustration adapter"""
    illustration = IIllustration(context, None)
    if illustration and illustration.has_data() and (illustration.renderer != HIDDEN_RENDERER_NAME):
        return illustration


@adapter_config(context=(ILinkIllustrationTarget, IPyAMSLayer),
                provides=IContentNavigationIllustration)
def link_content_navigation_illustration_factory(context, request):
    """Content navigation illustration adapter for basic link illustration targets"""
    illustration = ILinkIllustration(context, None)
    if illustration and illustration.has_data():
        return illustration


@adapter_config(context=(ISiteRoot, IPyAMSLayer), provides=IContentHeaderIllustration)
@adapter_config(context=(IBaseContent, IPyAMSLayer), provides=IContentHeaderIllustration)
def base_content_header_illustration_factory(context, request):
    """Base content header illustration adapter"""
    return None


@adapter_config(context=(ISharedContent, IPyAMSLayer), provides=IContentNavigationIllustration)
def shared_content_illustration_factory(context, request):
    """Shared content illustration factory"""
    version = context.visible_version
    if version is not None:
        return request.registry.queryMultiAdapter((version, request),
                                                  IContentNavigationIllustration)


@adapter_config(name='pyams_illustration',
                context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class PyAMSIllustrationTALESExtension(ContextRequestViewAdapter):
    """PyAMS illustration TALES extension"""

    def render(self, context=None, name=''):
        if context is None:
            context = self.context
        return self.request.registry.queryMultiAdapter((context, self.request),
                                                       IContentNavigationIllustration,
                                                       name=name)


@adapter_config(name='pyams_banner_illustration',
                context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class PyAMSBannerIllustrationTALESExtension(ContextRequestViewAdapter):
    """PyAMS banner illustration TALES extension"""

    def render(self, context=None, name=''):
        if context is None:
            context = self.context
        return self.request.registry.queryMultiAdapter((context, self.request),
                                                       IContentBannerIllustration,
                                                       name=name)


@adapter_config(name='pyams_header_illustration',
                context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class PyAMSHeaderIllustrationTALESExtension(ContextRequestViewAdapter):
    """PyAMS header illustration TALES extension"""

    def render(self, context=None, name=''):
        if context is None:
            context = self.context
        return self.request.registry.queryMultiAdapter((context, self.request),
                                                       IContentHeaderIllustration,
                                                       name=name)


#
# Illustration renderer with zoom settings
#

ILLUSTRATION_ZOOM_RENDERER_SETTINGS_KEY = 'pyams_content.illustration.renderer:zoom'


@factory_config(IIllustrationWithZoomSettings)
class IllustrationZoomSettings(Persistent, Location):
    """Illustration zoom renderer settings"""

    zoom_on_click = FieldProperty(IIllustrationWithZoomSettings['zoom_on_click'])


@adapter_config(context=IIllustration, provides=IIllustrationWithZoomSettings)
def illustration_with_zoom_settings_factory(context):
    """Illustration zoom renderer settings factory"""
    return get_annotation_adapter(context, ILLUSTRATION_ZOOM_RENDERER_SETTINGS_KEY,
                                  IIllustrationWithZoomSettings)


#
# Illustration renderers
#

@implementer(IIllustrationRenderer)
class BaseIllustrationRenderer(BaseContentRenderer):
    """Base illustration renderer"""

    context_attrs = ('author',)
    i18n_context_attrs = ('title', 'alt_title', 'description', 'data')

    position = None


@adapter_config(name='centered-before-body', context=(IIllustration, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/illustration-default.pt', layer=IPyAMSLayer)
class CenteredIllustrationRenderer(BaseIllustrationRenderer):
    """Centered illustration renderer"""

    label = _("Centered illustration before text")
    weight = 1

    position = ILLUSTRATION_BEFORE_BODY


@adapter_config(name='left+zoom', context=(IIllustration, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/illustration-left.pt', layer=IPyAMSLayer)
class LeftIllustrationWithZoomRenderer(BaseIllustrationRenderer):
    """Illustration renderer with small image and zoom"""

    label = _("Small illustration on the left")
    weight = 2

    position = ILLUSTRATION_BEFORE_BODY
    settings_interface = IIllustrationWithZoomSettings

    resources = (lightbox,)


@adapter_config(name='right+zoom', context=(IIllustration, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/illustration-right.pt', layer=IPyAMSLayer)
class RightIllustrationWithZoomRenderer(BaseIllustrationRenderer):
    """Illustration renderer with small image and zoom"""

    label = _("Small illustration on the right")
    weight = 3

    position = ILLUSTRATION_BEFORE_BODY
    settings_interface = IIllustrationWithZoomSettings

    resources = (lightbox,)


@adapter_config(name='default', context=(IIllustration, IPyAMSLayer), provides=IContentRenderer)
@template_config(template='templates/illustration-default.pt', layer=IPyAMSLayer)
class DefaultIllustrationRenderer(BaseIllustrationRenderer):
    """Default illustration renderer"""

    label = _("Centered illustration after text")
    weight = 4

    position = ILLUSTRATION_AFTER_BODY
