#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""
from datetime import date, timedelta

import iso8601
from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import Interface
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.calendar import ICalendarInfo
from pyams_content.features.search.portlet import ISearchResultsPortletSettings
from pyams_content.shared.calendar.interfaces import ICalendarManager
from pyams_default_theme.component.calendar.portlet.interfaces import IViewItemsPortletCalendarListRendererSettings
from pyams_default_theme.features.search.portlet import SearchResultsPortletBaseRenderer
from pyams_default_theme.shared.view.portlet import BaseViewItemsPortletRenderer
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.date import format_date, format_time
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility


__docformat__ = 'restructuredtext'

from pyams_default_theme import _
from pyams_utils.timezone import tztime


MONTH_FORMAT = _("%Y, %B")
DATE_FORMAT = _("%d")
TIME_FORMAT = _("%H:%M")
DATETIME_FORMAT = _("on {date}, at {time}")


class BaseCalendarItemsPortletRenderer(SearchResultsPortletBaseRenderer):
    """Base calendar items portlet renderer"""

    def __new__(cls, context, request, view, settings):
        calendar = query_utility(ICalendarManager)
        if calendar is None:
            return None
        return BaseViewItemsPortletRenderer.__new__(cls)

    def update(self):
        request = self.request
        settings = self.renderer_settings
        from_date = request.params.get('from_date')
        if from_date is None:
            request.GET['from_date'] = (date.today() - timedelta(days=settings.previous_months * 30)).isoformat()
        to_date = request.params.get('to_date')
        if to_date is None:
            request.GET['to_date'] = (date.today() + timedelta(days=settings.displayed_months * 30)).isoformat()
        #
        # Don't call super().update to avoid access to pagination settings
        # which are not required for calendar views!!
        super(SearchResultsPortletBaseRenderer, self).update()

    def get_sorted_dates(self, items):
        result = {}
        params = self.request.params
        from_date = iso8601.parse_date(params.get('from_date'))
        to_date = iso8601.parse_date(params.get('to_date'))
        for item in items:
            calendar = ICalendarInfo(item, None)
            if calendar is None:
                continue
            for event_date in calendar.calendar:
                if (event_date < from_date) or (event_date > to_date):
                    continue
                result.setdefault(event_date.date(), []).append((tztime(event_date).time(), item))
        yield from sorted(result.items())

    def get_location(self, item):
        calendar_info = ICalendarInfo(item, None)
        if calendar_info is not None:
            return II18n(calendar_info).query_attribute('location', request=self.request)
        return None

    def format_month(self, value):
        return format_date(value, MONTH_FORMAT, self.request)

    def format_time(self, date, time):
        translate = self.request.localizer.translate
        return translate(DATETIME_FORMAT).format(date=format_date(date, DATE_FORMAT, self.request),
                                                 time=format_time(time, TIME_FORMAT, self.request))


@factory_config(IViewItemsPortletCalendarListRendererSettings)
class ViewItemsPortletCalendarListRendererSettings(Persistent, Contained):
    """View items portlet calendar renderer settings"""

    previous_months = FieldProperty(IViewItemsPortletCalendarListRendererSettings['previous_months'])
    displayed_months = FieldProperty(IViewItemsPortletCalendarListRendererSettings['displayed_months'])


@adapter_config(name='calendar-list',
                required=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/calendar-list.pt', layer=IPyAMSLayer)
class ViewItemsPortletCalendarListRenderer(BaseCalendarItemsPortletRenderer):
    """View items portlet calendar list renderer"""

    label = _("Calendar: simple events list")
    weight = 30

    settings_interface = IViewItemsPortletCalendarListRendererSettings
