#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.decorator import reify
from zope.interface import Interface

from pyams_content.component.association.interfaces import IAssociationContainer, IAssociationInfo
from pyams_content.component.extfile.interfaces import IExtFile, IExtFileManagerInfo
from pyams_content.component.links.interfaces import IBaseLink
from pyams_content.root import ISiteRootConfiguration
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.configuration import IConfiguration
from pyams_skin.layer import IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_viewlet.viewlet import ViewContentProvider, contentprovider_config


@contentprovider_config(name='pyams.associations',
                        layer=IPyAMSUserLayer, view=Interface)
@template_config(template='templates/association-viewlet.pt',
                 layer=IPyAMSUserLayer)
class AssociationContentProvider(ViewContentProvider):
    """Association default content provider"""

    description_format = 'text'
    template_name = ''

    links = None
    attachments = None
    state = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.attachments = []
        self.links = []
        self.state = {}

    def get_associations(self):
        yield from IAssociationContainer(self.context).get_visible_items(self.request)

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)

    @reify
    def public_url(self):
        configuration = IConfiguration(self.request.root, None)
        if not ISiteRootConfiguration.providedBy(configuration):
            raise ''
        return configuration.public_url

    def update(self, settings=None, template_name='', **state):
        super().update()
        if settings is not None:
            self.description_format = settings.description_format
        self.template_name = template_name
        self.state.update(state)
        for item in self.get_associations():
            if IExtFile.providedBy(item):
                self.attachments.append(item)
            elif IBaseLink.providedBy(item):
                self.links.append(item)

    def render(self, template_name=''):
        return super().render(self.template_name)

    @property
    def attachments_title_prefix(self):
        manager_info = IExtFileManagerInfo(self.request.root, None)
        if manager_info is not None:
            return II18n(manager_info).query_attribute('default_title_prefix', request=self.request)
