#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from itertools import islice

from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.association.interfaces import IAssociationContainer, \
    IAssociationInfo
from pyams_content.component.association.interfaces.paragraph import IAssociationParagraph
from pyams_content.component.extfile.interfaces import IExtFile
from pyams_content.component.links.interfaces import IBaseLink, IInternalLink
from pyams_content.component.paragraph.interfaces import IParagraphContainer, \
    IParagraphContainerTarget
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_default_theme.component.association.interfaces import \
    IAssociationParagraphDefaultRendererSettings, IAssociationParagraphRemoteContentRendererSettings, \
    IAssociationParagraphSlicedRemoteContentRendererSettings
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


#
# Associations paragraph default renderer
#

ASSOCIATION_PARAGRAPH_DEFAULT_RENDERER_SETTINGS_KEY = 'pyams_content.association.renderer:default'


@factory_config(IAssociationParagraphDefaultRendererSettings)
class AssociationParagraphDefaultRendererSettings(Persistent, Location):
    """Association paragraph default renderer settings"""

    description_format = FieldProperty(IAssociationParagraphDefaultRendererSettings['description_format'])


@adapter_config(context=IAssociationParagraph,
                provides=IAssociationParagraphDefaultRendererSettings)
def association_paragraph_default_renderer_settings_factory(context):
    """Associations paragraph default renderer settings factory"""
    return get_annotation_adapter(context, ASSOCIATION_PARAGRAPH_DEFAULT_RENDERER_SETTINGS_KEY,
                                  IAssociationParagraphDefaultRendererSettings)


@adapter_config(name='default',
                context=(IAssociationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/association-default.pt',
                 layer=IPyAMSLayer)
class AssociationParagraphDefaultRenderer(BaseContentRenderer):
    """Associations paragraph default renderer"""

    label = _("Default associations renderer")
    i18n_context_attrs = ('title',)

    settings_interface = IAssociationParagraphDefaultRendererSettings

    def __init__(self, *args, **kwargs):
        super(AssociationParagraphDefaultRenderer, self).__init__(*args, **kwargs)
        self.attachments = []
        self.links = []

    def get_associations(self):
        yield from IAssociationContainer(self.context).get_visible_items(self.request)

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)

    def update(self):
        super().update()
        for item in self.get_associations():
            if IExtFile.providedBy(item):
                self.attachments.append(item)
            elif IBaseLink.providedBy(item):
                self.links.append(item)


#
# Associations paragraph remote content renderer
#

ASSOCIATION_PARAGRAPH_SLICED_REMOTE_CONTENT_RENDERER_SETTINGS_KEY = \
    'pyams_content.association.renderer:sliced-remote-content'
ASSOCIATION_PARAGRAPH_REMOTE_CONTENT_RENDERER_SETTINGS_KEY = \
    'pyams_content.association.renderer:remote-content'


@factory_config(IAssociationParagraphSlicedRemoteContentRendererSettings)
class AssociationParagraphSlicedRemoteContentRendererSettings(Persistent, Location):
    """Associations paragraph sliced remote content renderer settings"""

    display_title = FieldProperty(IAssociationParagraphSlicedRemoteContentRendererSettings['display_title'])
    display_header = FieldProperty(IAssociationParagraphSlicedRemoteContentRendererSettings['display_header'])
    paragraphs_count = FieldProperty(IAssociationParagraphSlicedRemoteContentRendererSettings['paragraphs_count'])


@adapter_config(context=IAssociationParagraph,
                provides=IAssociationParagraphSlicedRemoteContentRendererSettings)
def association_paragraph_sliced_remote_content_renderer_settings_factory(context):
    """Associations paragraph sliced remote content renderer settings factory"""
    return get_annotation_adapter(context, ASSOCIATION_PARAGRAPH_SLICED_REMOTE_CONTENT_RENDERER_SETTINGS_KEY,
                                  IAssociationParagraphSlicedRemoteContentRendererSettings)


@adapter_config(name='reportage',
                context=(IAssociationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/association-remote-content.pt',
                 layer=IPyAMSLayer)
class AssociationParagraphSlicedRemoteContentRenderer(BaseContentRenderer):
    """Associations container sliced remote content renderer"""

    label = _("Include first remote content paragraphs")
    weight = 100

    i18n_context_attrs = ('title',)
    links = ()

    settings_interface = IAssociationParagraphSlicedRemoteContentRendererSettings

    def update(self):
        super(AssociationParagraphSlicedRemoteContentRenderer, self).update()
        self.links = [item for item in IAssociationContainer(self.context).get_visible_items(self.request)
                      if IInternalLink.providedBy(item) and IParagraphContainerTarget.providedBy(item.target)]

    def get_renderers(self, target):
        container = IParagraphContainer(target, None)
        if container is not None:
            for renderer in islice(map(lambda x: x.get_renderer(self.request),
                                       container.get_visible_paragraphs()),
                                   self.settings.paragraphs_count):
                if renderer is not None:
                    renderer.update()
                    yield renderer.render()


@factory_config(IAssociationParagraphRemoteContentRendererSettings)
class AssociationParagraphRemoteContentRendererSettings(Persistent, Location):
    """Associations paragraph remote content renderer settings"""

    display_title = FieldProperty(IAssociationParagraphRemoteContentRendererSettings['display_title'])
    display_header = FieldProperty(IAssociationParagraphRemoteContentRendererSettings['display_header'])
    factories = FieldProperty(IAssociationParagraphRemoteContentRendererSettings['factories'])
    anchors_only = FieldProperty(IAssociationParagraphRemoteContentRendererSettings['anchors_only'])


@adapter_config(context=IAssociationParagraph,
                provides=IAssociationParagraphRemoteContentRendererSettings)
def association_paragraph_remote_content_renderer_settings_factory(context):
    """Associations paragraph remote content renderer settings factory"""
    return get_annotation_adapter(context, ASSOCIATION_PARAGRAPH_REMOTE_CONTENT_RENDERER_SETTINGS_KEY,
                                  IAssociationParagraphRemoteContentRendererSettings)


@adapter_config(name='remote-content',
                context=(IAssociationParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/association-remote-content.pt',
                 layer=IPyAMSLayer)
class AssociationParagraphRemoteContentRenderer(BaseContentRenderer):
    """Associations container remote content renderer"""

    label = _("Include remote content paragraphs")
    weight = 110

    i18n_context_attrs = ('title',)
    links = ()

    settings_interface = IAssociationParagraphRemoteContentRendererSettings

    def update(self):
        super(AssociationParagraphRemoteContentRenderer, self).update()
        self.links = [item for item in IAssociationContainer(self.context).get_visible_items(self.request)
                      if IInternalLink.providedBy(item) and IParagraphContainerTarget.providedBy(item.target)]

    def get_renderers(self, target):
        container = IParagraphContainer(target, None)
        if container is not None:
            settings = self.settings
            for renderer in map(lambda x: x.get_renderer(self.request),
                                container.get_visible_paragraphs(
                                    names=None,
                                    anchors_only=settings.anchors_only,
                                    factories=settings.factories)):
                if renderer is not None:
                    renderer.update()
                    yield renderer.render()
