#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import locale

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Any, NotEq, Or

from pyams_catalog.query import CatalogResultSet
from pyams_content.shared.common.interfaces import ISharedSite, ISharedTool
from pyams_content.zmi.interfaces import IUserAddingsMenuLabel
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.viewlet import ITopLinksViewletManager
from pyams_skin.viewlet.toplinks import TopLinksMenu, TopLinksViewlet
from pyams_template.template import template_config
from pyams_utils.list import unique
from pyams_utils.registry import get_all_utilities_registered_for, get_utilities_for, get_utility
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='shared-sites.menu', layer=IAdminLayer, manager=ITopLinksViewletManager,
                weight=20)
class SharedSitesMenu(TopLinksViewlet):
    """Shared sites menu"""

    label = ''
    css_class = 'top-menu bordered margin-top-10'
    dropdown_label = _("Shared sites")

    def update(self):
        super(SharedSitesMenu, self).update()
        context = self.context
        request = self.request
        parent = self.__parent__
        for site in sorted(get_all_utilities_registered_for(ISharedSite),
                           key=lambda x: locale.strxfrm(II18n(x).query_attribute('title',
                                                                                 request=request)
                                                        or '')):
            menu = TopLinksMenu(context, request, parent, self)
            menu.label = II18n(site).query_attribute('title', request=request) or site.__name__
            menu.url = absolute_url(site, request, 'admin#dashboard.html')
            self.viewlets.append(menu)


@viewlet_config(name='shared-contents.menu', layer=IAdminLayer, manager=ITopLinksViewletManager,
                weight=30)
class SharedContentsMenu(TopLinksViewlet):
    """Shared contents menu"""

    label = ''
    css_class = 'top-menu bordered margin-top-10'
    dropdown_label = _("Shared contents")

    def update(self):
        super(SharedContentsMenu, self).update()
        context = self.context
        request = self.request
        parent = self.__parent__
        for name, tool in sorted(get_utilities_for(ISharedTool),
                                 key=lambda x: locale.strxfrm(II18n(x[1]).query_attribute('title',
                                                                                          request=request)
                                                              or '')):
            if (not name) or (not tool.shared_content_menu):
                continue
            menu = TopLinksMenu(context, request, parent, self)
            menu.label = II18n(tool).query_attribute('title', request=request) or tool.__name__
            menu.url = absolute_url(tool, request, 'admin#dashboard.html')
            self.viewlets.append(menu)


@viewlet_config(name='shared-tools.menu', layer=IAdminLayer, manager=ITopLinksViewletManager,
                weight=40)
class SharedToolsMenu(TopLinksViewlet):
    """Shared tools menu"""

    label = ''
    css_class = 'top-menu bordered margin-top-10'
    dropdown_label = _("Shared tools")

    def update(self):
        super(SharedToolsMenu, self).update()
        context = self.context
        request = self.request
        parent = self.__parent__
        for name, tool in sorted(get_utilities_for(ISharedTool),
                                 key=lambda x: locale.strxfrm(II18n(x[1]).query_attribute('title',
                                                                                          request=request)
                                                              or '')):
            if (not name) or tool.shared_content_menu:
                continue
            menu = TopLinksMenu(context, request, parent, self)
            menu.label = II18n(tool).query_attribute('title', request=request) or tool.__name__
            menu.url = absolute_url(tool, request, 'admin#dashboard.html')
            self.viewlets.append(menu)


@viewlet_config(name='user-roles.menu', layer=IAdminLayer, manager=ITopLinksViewletManager,
                weight=90)
class UserRolesMenu(TopLinksViewlet):
    """User roles menu"""

    label = ''
    css_class = 'top-menu bordered margin-top-10'
    dropdown_label = _("My roles")

    def update(self):
        super(UserRolesMenu, self).update()
        catalog = get_utility(ICatalog)
        context = self.context
        request = self.request
        parent = self.__parent__
        params = And(Or(Any(catalog['role:contributor'], {request.principal.id}),
                        Any(catalog['role:manager'], {request.principal.id}),
                        Any(catalog['role:pilot'], {request.principal.id})),
                     NotEq(catalog['content_type'], None))
        for tool in sorted(unique(CatalogResultSet(CatalogQuery(catalog).query(params))),
                           key=lambda x: locale.strxfrm(II18n(x).query_attribute('title',
                                                                                 request=request)
                                                        or '')):
            menu = TopLinksMenu(context, request, parent, self)
            menu.label = II18n(tool).query_attribute('title', request=request) or tool.__name__
            menu.url = absolute_url(tool, request, 'admin#dashboard.html')
            self.viewlets.append(menu)


@viewlet_config(name='user-addings.menu', layer=IAdminLayer, manager=ITopLinksViewletManager,
                weight=95)
@template_config(template='templates/user-addings.pt', layer=IAdminLayer)
class UserAddingsMenu(TopLinksViewlet):
    """User addings menu"""

    label = ''
    css_class = 'top-menu margin-top-5-neg'
    dropdown_label = ''

    def update(self):
        super(UserAddingsMenu, self).update()
        catalog = get_utility(ICatalog)
        context = self.context
        request = self.request
        parent = self.__parent__
        registry = request.registry
        params = And(Any(catalog['role:contributor'], {request.principal.id}),
                     NotEq(catalog['content_type'], None))
        for tool in sorted(unique(CatalogResultSet(CatalogQuery(catalog).query(params))),
                           key=lambda x: locale.strxfrm(II18n(x).query_attribute('title',
                                                                                 request=request)
                                                        or '')):
            if not ISharedTool.providedBy(tool):
                continue
            content_class = tool.shared_content_factory.content_class
            if content_class is None:
                continue
            menu = TopLinksMenu(context, request, parent, self)
            adapter = registry.queryMultiAdapter((tool, request), IUserAddingsMenuLabel)
            if adapter is None:
                menu.label = request.localizer.translate(content_class.content_name)
            else:
                menu.label = adapter.label
            menu.url = absolute_url(tool, request, 'add-shared-content.html')
            menu.data = {'data-toggle': 'modal'}
            self.viewlets.append(menu)
