(function($, globals) {

	"use strict";

	const MyAMS = globals.MyAMS;


	/**
	 * VueJS calendar component
	 *
	 * Create a small component based on JQuery datetimepicker to be able to select
	 * multiple datetimes
	 *
	 * @param element: source element
	 * @param name: input name
	 * @param values: initial values
	 * @returns {*|boolean}
	 */
	function createCalendar(element, legend, name, values) {
		MyAMS.ajax.check(window.Vue,
						 MyAMS.baseURL + '/ext/vue' + MyAMS.devext + '.js',
						 function() {
							 return new Vue({
								 el: element,
								 data: () => {
									 return {
										 now: new Date(),
										 legend: legend,
										 name: name,
										 dates: values
									 }
								 },
								 template: `
									<fieldset class="col-12 col-sm-6 col-md-4">
										<legend class="inner no-padding">
											{{ legend }}
										</legend>
										<input type="hidden"
											   v-bind:name="name"
											   v-bind:value="values" />
										<div class="input">
											<i class="icon-append fa fa-plus text-primary"
											   @click="addSelectedDate" />
											<input class="datetimepicker"
												   @keyup.enter="addSelectedDate" 
												   v-once />
										</div>
										<div class="viewport-y viewport-200">
											<div class="padding-x-5 padding-y-2"
												 v-for="(date, index) in dates"
												 v-bind:class="{
													'text-dashed text-danger': date < now
												 }">
												{{ date.toLocaleString() }}
												<i class="fa fa-fw fa-trash text-primary opaque"
												   v-on:click="removeSelectedDate($event, index)"></i>
											</div>
										</div>
									</fieldset>
								`,
								 mounted: function() {
									 const self = this;
									 MyAMS.ajax.check($.fn.datetimepicker,
										 MyAMS.baseURL + 'ext/jquery-datetimepicker' + MyAMS.devext + '.js',
										 function() {
											 MyAMS.getCSS(MyAMS.baseURL + '../css/ext/jquery-datetimepicker' + MyAMS.devext + '.css',
												 'jquery-datetimepicker',
												 function() {
													 $('.datetimepicker', self.$el).datetimepicker({
														 lang: MyAMS.lang,
														 format: 'd/m/Y H:i',
														 datepicker: true,
														 dayOfWeekStart: 1,
														 timepicker: true,
														 closeOnDateSelect: false,
														 closeOnTimeSelect: true
													 });
												 });
										 });
								 },
								 computed: {
									 values() {
										 return this.dates.map(x => x.toISOString()).join(';');
									 }
								 },
								 methods: {
									 isOver(date) {
										 console.debug(date, this.now);
										 return date < this.now;
									 },
									 addSelectedDate() {
										 const
											 picker = $('.datetimepicker', this.$el),
											 value = picker.val();
										 if (value) {
											 const
												 [date, time] = value.split(' '),
												 [day, month, year] = date.split('/'),
												 [hour, minutes] = time.split(':'),
												 newDate = new Date(year, month - 1, day, hour, minutes);
											 const oldDate = this.dates.find(date => date.getTime() === newDate.getTime());
											 if (oldDate === undefined) {
												 this.dates.push(newDate);
												 this.dates.sort((a, b) => {
													 return a.toISOString() > b.toISOString() ? -1 : 1;
												 });
											 }
										 }
									 },
									 removeSelectedDate(event, index) {
										 this.dates.splice(index, 1);
									 }
								 },
								 beforeDestroy: function() {
									 $(this.$el).datetimepicker('hide').datetimepicker('destroy');
								 }
							 });
						 });
	}


	const PyAMS_content = {

		/**
		 * Custom widgets management
		 */
		widget: {

			/** Treeview widget */
			treeview: {

				selectFolder: function(event, node) {
					var target = $(event.target);
					target.siblings('input[type="hidden"]').val(node.id);
				},

				unselectFolder: function(event, node) {
					var target = $(event.target);
					target.siblings('input[type="hidden"]').val(null);
				}
			}
		},

		/**
		 * TinyMCE editor configuration
		 */
		TinyMCE: {

			initEditor: function(settings) {

				// Update TinyMCE translations
				tinyMCE.addI18n('fr', {
					"Link list": "Liste de liens",
					"Toggle h3 header": "En-tête H3",
					"Toggle h4 header": "En-tête H4",
					"Insert internal link": "Insérer un lien interne",
					"Link title": "Texte à afficher",
					"Internal number": "N° interne",
					"Custom styles": "Styles spécifiques"
				});

				// Declare internal link input plug-in
				tinymce.PluginManager.add('internal_links', function(editor, url) {
					editor.addButton('internal_links', {
						icon: 'cloud-check',
						tooltip: "Insert internal link",
						image: '/--static--/pyams_content/img/internal-link.png',
						stateSelector: 'a[href^="oid://"]',
						onclick: function() {
							var data = {},
								dom = editor.dom,
								selection = editor.selection,
								selectedElm = selection.getNode(),
								anchorElm = dom.getParent(selectedElm, 'a[href]');
							data.text = anchorElm ? (anchorElm.innerText || anchorElm.textContent) : selection.getContent({format: 'text'});
							data.href = anchorElm ? dom.getAttrib(anchorElm, 'href') : '';
							editor.windowManager.open({
								title: "Insert internal link",
								body: [
									{
										type: 'textbox',
										name: 'oid',
										label:'Internal number',
										value: data.href,
										onchange: function() {
											data.href = this.value();
										}
									}, {
										type: 'textbox',
										name: 'title',
										label: 'Link title',
										value: data.text,
										onchange: function() {
											data.text = this.value();
										}
									}
								],
								onsubmit: function(e) {
									editor.focus();  // activate change events
									var href = data.href;
									if (href) {
										if (anchorElm) {
											if ('innerText' in anchorElm) {
												anchorElm.innerText = data.text;
											} else {
												anchorElm.textContent = data.text;
											}
											dom.setAttribs(anchorElm, {href: data.href});
											selection.select(anchorElm);
											editor.undoManager.add();
										} else {
											editor.insertContent('<a href="oid://' + e.data.oid + '">' + e.data.title + '</a>');
										}
									} else {
										editor.execCommand('unlink');
									}
								}
							});
						}
					})
				});

				// Declare direct H3 and H4 style formatter
				tinyMCE.PluginManager.add('headers', function (editor, url) {
					['h3', 'h4'].forEach(function (name) {
						editor.addButton("header-" + name, {
							tooltip: "Toggle " + name + " header",
							text: name.toUpperCase(),
							onClick: function () {
								editor.execCommand('mceToggleFormat', false, name);
							},
							onPostRender: function () {
								var self = this,
									setup = function () {
										editor.formatter.formatChanged(name, function (state) {
											self.active(state);
										});
									};
								editor.formatter ? setup() : editor.on('init', setup);
							}
						})
					});
				});

				// Update TinyMCE settings
				settings.paste_as_text = true;
				settings.image_list = PyAMS_content.TinyMCE.getImagesList;
				settings.link_list = PyAMS_content.TinyMCE.getLinksList;
				settings.style_formats = [
					{
						title: 'Inline',
						items: [
							{title: 'Bold', icon: 'bold', format: 'bold'},
							{title: 'Italic', icon: 'italic', format: 'italic'},
							{title: 'Underline', icon: 'underline', format: 'underline'},
							{title: 'Strikethrough', icon: 'strikethrough', format: 'strikethrough'},
							{title: 'Superscript', icon: 'superscript', format: 'superscript'},
							{title: 'Subscript', icon: 'subscript', format: 'subscript'},
							{title: 'Code', icon: 'code', format: 'code'}
						]
					},
					{
						title: 'Blocks',
						items: [
							{title: 'Paragraph', format: 'p'},
							{title: 'Blockquote', format: 'blockquote'},
							{title: 'Div', format: 'div'},
							{title: 'Pre', format: 'pre'}
						]
					},
					{
						title: 'Alignment',
						items: [
							{title: 'Left', icon: 'alignleft', format: 'alignleft'},
							{title: 'Center', icon: 'aligncenter', format: 'aligncenter'},
							{title: 'Right', icon: 'alignright', format: 'alignright'}
						]
					}
				];

				// Get custom editor styles
				var editor_data = this.data();
				if (editor_data.amsTinymceEditorStyles) {
					settings.style_formats.push({
						title: 'Custom styles',
						items: editor_data.amsTinymceEditorStyles
					});
				}

				// Define plugins
				settings.plugins += ' internal_links headers';
				if (settings.toolbar1) {
					settings.toolbar1 = "undo redo | pastetext | header-h3 header-h4 styleselect | bold italic | " +
										"alignleft aligncenter alignright | " +
										"bullist numlist outdent indent";
				}
				if (settings.toolbar2) {
					settings.toolbar2 = "forecolor backcolor | charmap internal_links link | " +
										"fullscreen preview print | code";
				}
				return settings;
			},

			getImagesList: function(callback) {
				var form = $(document.activeElement).parents('form');
				if (form.exists()) {
					var base = form.attr('data-ams-form-handler') || form.attr('action');
					var target = base.substr(0, base.lastIndexOf('/') + 1);
					return MyAMS.ajax.post(target + 'get-images-list.json', {}, callback);
				}
			},

			getLinksList: function(callback) {
				var form = $(document.activeElement).parents('form');
				if (form.exists()) {
					var base = form.attr('data-ams-form-handler') || form.attr('action');
					var target = base.substr(0, base.lastIndexOf('/')+1);
					return MyAMS.ajax.post(target + 'get-links-list.json', {}, callback);
				}
			}
		},

		/**
		 * Galleries management
		 */
		galleries: {

			addMediaCallback: function(options) {
				var gallery = $('.gallery', $('[id="' + options.parent + '"]'));
				var switcher = gallery.siblings('.switcher');
				if (switcher.parents('fieldset:first').hasClass('switched')) {
					switcher.click();
				}
				for (var i=0; i < options.medias.length; i++) {
					gallery.append(options.medias[i]);
				}
				gallery.sortable({
					helper: 'clone',
					stop: PyAMS_content.galleries.setOrder
				})
			},

			updateMediaTitle: function(options) {
				$('img[id="' + options.media_id + '"]').attr('original-title', options.title);
			},

			switchMediaVisibility: function(element) {
				return function() {
					var source = $(this);
					var media = source.parents('.media');
					var gallery = media.parents('.gallery');
					$('i', source).attr('class', 'fa fa-fw fa-spinner fa-pulse');
					MyAMS.ajax.post(gallery.data('ams-location') + '/set-media-visibility.json',
									{object_name: media.data('ams-element-name')},
									function(result, status) {
										if (result.visible) {
											$('i', source).attr('class', 'fa fa-fw fa-eye');
											source.parents('.btn-group').siblings('a.fancyimg').removeClass('not-visible');
										} else {
											$('i', source).attr('class', 'fa fa-fw fa-eye-slash text-danger');
											source.parents('.btn-group').siblings('a.fancyimg').addClass('not-visible');
										}
										if (result.handle_json) {
											MyAMS.ajax.handleJSON(result);
										}
									});
				};
			},

			setOrder: function(event, ui) {
				if (ui && ui.item.hasClass('already-dropped')) {
					return;
				}
				var gallery = ui.item.parents('.gallery');
				var ids = $('.media', gallery).listattr('data-ams-element-name');
				MyAMS.ajax.post(gallery.data('ams-location') + '/set-medias-order.json',
								{medias: JSON.stringify(ids)});
			},

			removeMedia: function(element) {
				return function() {
					var link = $(this);
					MyAMS.skin.bigBox({
						title: MyAMS.i18n.WARNING,
						content: '<i class="text-danger fa fa-2x fa-bell shake animated"></i>&nbsp; ' + MyAMS.i18n.DELETE_WARNING,
						buttons: MyAMS.i18n.BTN_OK_CANCEL
					}, function(button) {
						if (button === MyAMS.i18n.BTN_OK) {
							var gallery = link.parents('.gallery');
							var location = gallery.data('ams-location');
							var media = link.parents('.media');
							var object_name = media.data('ams-element-name');
							MyAMS.ajax.post(location + '/delete-media.json',
											{object_name: object_name},
											function(result, status) {
												media.remove();
												if (result.handle_json) {
													MyAMS.ajax.handleJSON(result);
												}
											});
						}
					});
				};
			},

			afterFancyboxLoad: function(current, previous) {
				var element = current.element;
				if (element.hasClass('not-visible')) {
					current.inner.prepend('<div class="hidden-mask"></div>');
				}
			}
		},

		/**
		 * Illustrations management
		 */
		illustration: {

			addIllustration: function() {
				var source = $(this);
				var group = source.parents('.btn-group');
				var legend = group.siblings('legend.switcher');
				$('i.fa-plus', legend).click();
				source.hide();
				$('input[type="file"]', legend.siblings('.form-group:first')).click();
			}
		},

		/**
		 * Paragraphs management
		 */
		paragraphs: {

			/* Store and restore paragraphs state on reload */
			preReload: function() {
				PyAMS_content.paragraphs.switched = $('i.switch.fa-minus-square-o', '#paragraphs_list').parents('tr').listattr('id');
			},

			postReload: function() {
				$(PyAMS_content.paragraphs.switched).each(function() {
					$('i.switch.fa-plus-square-o', '[id="' + this + '"]').parents('div').first().click();
				});
				delete PyAMS_content.paragraphs.switched;
			},

			refreshParagraph: function(changes) {
				var para = $('tr[id="' + changes.object_id + '"]');
				$('span.title', para).html(changes.title || ' - - - - - - - -');
			},

			switchEditor: function(element) {
				var source = $(this);
				var switcher = $('i.switch', source);
				var td = source.parents('td');
				var editor = $('.editor', td);
				var para = source.parents('tr');
				if (switcher.hasClass('fa-plus-square-o')) {
					var container = para.parents('table');
					editor.html('<h1 class="loading"><i class="fa fa-2x fa-gear fa-spin"></i></h1>');
					MyAMS.ajax.post(container.data('ams-location') + '/get-paragraph-editor.json',
									{object_name: para.data('ams-element-name')},
									function(result) {
										editor.html(result);
										if (result) {
											MyAMS.initContent(editor);
											switcher.removeClass('fa-plus-square-o')
													.addClass('fa-minus-square-o');
											para.data('ams-disabled-handlers', true);
											MyAMS.skin.scrollTo(editor, {
												offset: editor.height() - para.height()
											});
										}
									});
				} else {
					MyAMS.skin.cleanContainer(editor);
					editor.empty();
					switcher.removeClass('fa-minus-square-o')
							.addClass('fa-plus-square-o');
					para.removeData('ams-disabled-handlers');
				}
			},

			switchLastEditor: function(table_id) {
				var table = $('table[id="' + table_id + '"]');
				var tr = $('tr:last', table);
				$('[data-ams-click-handler="PyAMS_content.paragraphs.switchEditor"]', tr).click();
			},

			switchAllEditors: function(element) {
				var source = $(this);
				var switcher = $('i', source);
				var container = source.parents('table');
				if (switcher.hasClass('fa-plus-square-o')) {
					switcher.removeClass('fa-plus-square-o')
							.addClass('fa-cog fa-spin');
					MyAMS.ajax.post(container.data('ams-location') + '/get-paragraphs-editors.json',
									{},
									function(result) {
										for (var para_id in result) {
											if (!result.hasOwnProperty(para_id)) {
												continue;
											}
											var para = $('tr[data-ams-element-name="' + para_id + '"]', container);
											var editor = $('.editor', para);
											if (editor.is(':empty')) {
												editor.html(result[para_id]);
											}
											$('.fa-plus-square-o', para).removeClass('fa-plus-square-o')
																		.addClass('fa-minus-square-o');
											para.data('ams-disabled-handlers', true);
										}
										if (!$('i.fa-plus-square-o', $('tbody', container)).exists()) {
											switcher.removeClass('fa-cog fa-spin')
													.addClass('fa-minus-square-o');
										}
										MyAMS.initContent(container);
									});
				} else {
					$('.editor', container).each(function() {
						MyAMS.skin.cleanContainer($(this));
						$(this).empty();
					});
					$('.fa-minus-square-o', container).removeClass('fa-minus-square-o')
													  .addClass('fa-plus-square-o');
					$('tr', container).removeData('ams-disabled-handlers');
				}
			},

			updateToolbar: function(settings) {
				var para = $('tr[id="' + settings.object_id + '"]');
				var toolbar = $('.title-toolbar', para);
				toolbar.replaceWith(settings.toolbar_tag);
				toolbar = $('.title-toolbar', para);
				MyAMS.initContent(toolbar);
			},

			updateMarkers: function(settings) {
				var para = $('tr[id="' + settings.object_id + '"]');
				var toolbar = $('.title-toolbar', para);
				var marker = $('DIV.action.' + settings.marker_type, toolbar);
				if (marker.exists()) {
					marker.replaceWith(settings.marker_tag);
				} else {
					$(settings.marker_tag).appendTo(toolbar);
				}
				if (settings.marker_tag) {
					marker = $('DIV.action.' + settings.marker_type, toolbar);
					MyAMS.initContent(marker);
				}
				MyAMS.helpers.sort(toolbar, 'weight');
			}
		},

		/**
		 * Pictograms management
		 */
		pictograms: {

			initManagerSelection: function() {
				var form = $(this);
				var selected = $('input[type="hidden"]', $('.selected-pictograms', form)).listattr('value');
				return {
					selected: JSON.stringify(selected)
				};
			},

			switchPictogram: function() {
				var pictogram = $(this);
				var parent = pictogram.parents('.pictograms');
				var manager = parent.parents('.pictograms-manager');
				if (parent.hasClass('available-pictograms')) {
					$('.selected-pictograms', manager).append(pictogram);
				} else {
					$('.available-pictograms', manager).append(pictogram);
				}
			}
		},

		/**
		 * Themes management
		 */
		themes: {

			updateThesaurus: function(event) {
				var form = $(this).parents('form');
				// clear extract name
				var extract = $('select[name$=".extract_name:list"]', form);
				var plugin = extract.data('select2');
				extract.empty();
				extract.select2('data', null);
				plugin.results.empty();
				// check thesaurus for new extracts
				var thesaurus = $('select[name$=".thesaurus_name:list"]', form);
				var thesaurus_name = thesaurus.val();
				if (thesaurus_name) {
					MyAMS.jsonrpc.post('getExtracts',
									   {thesaurus_name: thesaurus_name},
									   {url: '/api/thesaurus/json'},
									   function(data) {
											extract.empty();
											$(data.result).each(function() {
												$('<option></option>').attr('value', this.id)
																	  .text(this.text)
																	  .appendTo(extract);
											});
											plugin.opts.populateResults.call(plugin, plugin.results, data.result,
																			 {term: ''});
									   });
				}
			}
		},

		/**
		 * Calendar component
		 */
		calendar: {

			init: function() {
				var calendar = this,
					$calendar = $(calendar),
					legend = $calendar.data('ams-legend'),
					name = $calendar.attr('name'),
					values = $calendar.val().split(';')
						.filter(x => x.length > 0)
						.map(x => new Date(x));
				createCalendar(calendar, legend, name, values);
			}
		},

		/**
		 * Image map management
		 */
		imgmap: {

			init: function() {
				var image = $(this);
				MyAMS.ajax.check($.fn.canvasAreaDraw,
								 '/--static--/pyams_content/js/jquery-canvasAreaDraw' + MyAMS.devext + '.js',
								 function() {
									image.canvasAreaDraw({imageUrl: image.data('ams-image-url')});
								 });
			},

			initPreview: function() {
				var image = $(this);
				MyAMS.ajax.check($.fn.mapster,
								 '/--static--/pyams_content/js/jquery-imagemapster-1.2.10' + MyAMS.devext + '.js',
								 function() {
									image.mapster({
										fillColor: 'ff0000',
										fillOpacity: 0.35,
										selected: true,
										highlight: true,
										staticState: true
									});
								 });
			}
		},

		/**
		 * Site management
		 */
		site: {

			switchVisibility: function() {
				return function() {
					var source = $(this);
					var content = source.parents('tr').first();
					MyAMS.ajax.post(content.data('ams-location') + '/switch-content-visibility.json',
									{object_name: content.data('ams-element-name')},
									function(result, status) {
										var klass = 'fa-eye';
										if (!result.visible) {
											klass += '-slash';
										}
										if (!result.published) {
											klass += ' text-danger';
										}
										$('i', source).attr('class', 'fa fa-fw ' + klass);
									});
				};
			}
		},

		/**
		 * Review comments management
		 */
		review: {

			timer: null,
			timer_duration: {
				general: 30000,
				chat: 5000
			},

			initComments: function(element) {
				var chat = $('.chat-body', element);
				chat.animate({scrollTop: chat[0].scrollHeight}, 1000);
				clearInterval(PyAMS_content.review.timer);
				PyAMS_content.review.timer = setInterval(PyAMS_content.review.updateComments,
														 PyAMS_content.review.timer_duration.chat);
				MyAMS.skin.registerCleanCallback(PyAMS_content.review.cleanCommentsCallback);
			},

			cleanCommentsCallback: function() {
				clearInterval(PyAMS_content.review.timer);
				PyAMS_content.review.timer = setInterval(PyAMS_content.review.updateComments,
														 PyAMS_content.review.timer_duration.general);
			},

			updateComments: function() {
				var badge = $('.badge', 'nav a[href="#review-comments.html"]'),
					count;
				var chat = $('.chat-body', '.widget-body');
				if (chat.exists()) {
					count = $('.message', chat).length;
				} else {
					count = parseInt(badge.text());
				}
				MyAMS.ajax.post('get-last-review-comments.json', {count: count}, function(result) {
					if (chat.exists()) {
						badge.removeClass('bg-color-danger')
							 .addClass('bg-color-info');
					}
					if (count !== result.count) {
						badge.text(result.count).removeClass('hidden');
						if (chat.exists()) {
							$('.messages', chat).append(result.content);
							chat.animate({scrollTop: chat[0].scrollHeight}, 1000);
						}
						if (!chat.exists()) {
							badge.removeClass('bg-color-info')
								 .addClass('bg-color-danger')
								 .animate({'padding': '3px 12px 2px',
										   'margin-right': '9px'}, 'slow', function() {
									$(this).animate({'padding': '3px 6px 2px',
													 'margin-right': '15px'}, 'slow');
								 });
						}
					}
				});
			},

			initCommentData: function(veto) {
				var chat = $('.chat-body', '.widget-body');
				return {count: $('.message', chat).length};
			},

			addCommentAction: function() {
				return function() {
					$('textarea[name="comment"]').focus();
				};
			},

			addCommentCallback: function(options) {
				var form = $(this);
				var widget = form.parents('.widget-body');
				$('.messages', widget).append(options.content);
				$('textarea[name="comment"]', form).val('');
				var chat = $('.chat-body', widget);
				chat.animate({scrollTop: chat[0].scrollHeight}, 1000);
				$('.badge', 'nav a[href="#review-comments.html"]').text(options.count).removeClass('hidden');
			}
		},

		/**
		 * Header management
		 */
		header: {

			submitEditForm: function() {
				var form = $(this).parents('form').first();
				MyAMS.form.submit(form, {form_data: {'autosubmit': true}});
			}
		},

		/**
		 * Footer management
		 */
		footer: {

			submitEditForm: function() {
				var form = $(this).parents('form').first();
				MyAMS.form.submit(form, {form_data: {'autosubmit': true}});
			}
		},

		/**
		 * User profile management
		 */
		profile: {

			switchFavorite: function() {
				var star = $(this);
				var oid = star.data('sequence-oid');
				MyAMS.ajax.post('switch-user-favorite.json',
								{oid: oid},
								function(result, status) {
									if (result.favorite) {
										star.removeClass('fa-star-o')
											.addClass('fa-star');
									} else {
										star.removeClass('fa-star')
											.addClass('fa-star-o');
									}
								});
			}
		}
	};

	var badge = $('.badge', 'nav a[href="#review-comments.html"]');
	if (badge.exists()) {
		PyAMS_content.review.timer = setInterval(PyAMS_content.review.updateComments,
												 PyAMS_content.review.timer_duration.general);
	}

	globals.PyAMS_content = PyAMS_content;

})(jQuery, this);
