#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

import logging
from datetime import datetime, timedelta
from itertools import islice, tee

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq, Gt, Lt, NotAny
from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.interface import implementer, provider
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent.interfaces import IObjectModifiedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_cache.beaker import get_cache
from pyams_catalog.query import CatalogResultSet, or_
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review.interfaces import IReviewTarget
from pyams_content.shared.common import IGNORED_CONTENT_TYPES, IWfSharedContent, \
    IWfSharedContentFactory, SharedContent, WfSharedContent, register_content_type
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_content.shared.view.interfaces import END_PARAMS_MARKER, IView, IViewQuery, \
    IViewQueryFilterExtension, IViewQueryParamsExtension, IViewSettings, IViewUserQuery, IWfView, IWfViewFactory, \
    VIEW_CONTENT_NAME, VIEW_CONTENT_TYPE
from pyams_content.interfaces import RELEVANCE_ORDER
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.list import unique_iter
from pyams_utils.registry import get_global_registry, get_utility
from pyams_utils.timezone import tztime
from pyams_workflow.interfaces import IWorkflow


logger = logging.getLogger("PyAMS (content)")

VIEWS_CACHE_REGION = 'views'
VIEWS_CACHE_NAME = 'PyAMS::view'

VIEW_CACHE_KEY = 'view_{view}'
VIEW_CONTEXT_CACHE_KEY = 'view_{view}.context_{context}'

_MARKER = object()


@implementer(IWfView, IPreviewTarget, IReviewTarget)
class WfView(WfSharedContent):
    """Base view"""

    content_type = VIEW_CONTENT_TYPE
    content_name = VIEW_CONTENT_NAME

    handle_content_url = False
    handle_header = False
    handle_description = False

    select_context_path = FieldProperty(IWfView['select_context_path'])
    select_context_type = FieldProperty(IWfView['select_context_type'])
    selected_content_types = FieldProperty(IWfView['selected_content_types'])
    select_context_datatype = FieldProperty(IWfView['select_context_datatype'])
    selected_datatypes = FieldProperty(IWfView['selected_datatypes'])
    excluded_content_types = FieldProperty(IWfView['excluded_content_types'])
    excluded_datatypes = FieldProperty(IWfView['excluded_datatypes'])
    order_by = FieldProperty(IWfView['order_by'])
    reversed_order = FieldProperty(IWfView['reversed_order'])
    limit = FieldProperty(IWfView['limit'])
    age_limit = FieldProperty(IWfView['age_limit'])

    @property
    def is_using_context(self):
        if self.select_context_path or self.select_context_type:
            return True
        registry = get_global_registry()
        for name, adapter in registry.getAdapters((self,), IViewSettings):
            if not name:
                continue
            if adapter.is_using_context:
                return True
        return False

    def get_content_path(self, context):
        if self.select_context_path:
            intids = get_utility(IIntIds)
            return intids.queryId(context)

    def get_content_types(self, context):
        content_types = set()
        if self.select_context_type and IWfSharedContent.providedBy(context):
            content_types.add(context.content_type)
        if self.selected_content_types:
            content_types |= set(self.selected_content_types)
        return list(content_types)

    def get_data_types(self, context):
        data_types = set()
        if self.select_context_datatype:
            content = IWfTypedSharedContent(context, None)
            if content is not None:
                data_types.add(content.data_type)
        if self.selected_datatypes:
            data_types |= set(self.selected_datatypes)
        return list(data_types)

    def get_excluded_content_types(self, context):
        return list(self.excluded_content_types or ())

    def get_excluded_data_types(self, context):
        return list(self.excluded_datatypes or ())

    def get_results(self, context, sort_index=None, reverse=None, limit=None,
                    start=0, length=999, ignore_cache=False, get_count=False, request=None,
                    aggregates=None, settings=None, **kwargs):
        results, count, aggregations = _MARKER, 0, {}
        if not ignore_cache:
            # check for cache
            views_cache = get_cache(VIEWS_CACHE_REGION, VIEWS_CACHE_NAME)
            if self.is_using_context:
                cache_key = VIEW_CONTEXT_CACHE_KEY.format(view=ICacheKeyValue(self),
                                                          context=ICacheKeyValue(context))
            else:
                cache_key = VIEW_CACHE_KEY.format(view=ICacheKeyValue(self))
            try:
                results = views_cache.get_value(cache_key)
                count = views_cache.get_value(cache_key + '::count')
                aggregations = views_cache.get_value(cache_key + '::aggregations')
            except KeyError:
                pass
        # Execute query
        if results is _MARKER:
            registry = get_current_registry()
            adapter = registry.getAdapter(self, IViewQuery)
            if not sort_index:
                sort_index = self.order_by
            # Get query results
            results, count, aggregations = adapter.get_results(context,
                                                               sort_index,
                                                               reverse if reverse is not None
                                                                   else self.reversed_order,
                                                               limit or self.limit,
                                                               request=request,
                                                               aggregates=aggregates,
                                                               settings=settings,
                                                               **kwargs)
            count = min(count, limit or self.limit or 999)
            cache, results = tee(islice(results, start, start + length))
            if not ignore_cache:
                intids = get_utility(IIntIds)
                views_cache.set_value(cache_key, [intids.queryId(item) for item in cache])
                views_cache.set_value(cache_key + '::count', count)
                views_cache.set_value(cache_key + '::aggregations', aggregations)
                logger.debug("Storing view items to cache key {0}".format(cache_key))
        else:
            results = CatalogResultSet(results)
            logger.debug("Retrieving view items from cache key {0}".format(cache_key))
        return (results, count, aggregations) if get_count else results


register_content_type(WfView, shared_content=False)


@provider(IWfViewFactory)
@implementer(IView)
class View(SharedContent):
    """Workflow managed view class"""


@adapter_config(context=IWfViewFactory, provides=IWfSharedContentFactory)
def view_content_factory(context):
    return WfView


@adapter_config(context=IWfView, provides=IViewQuery)
class ViewQuery(ContextAdapter):
    """View query"""

    def get_params(self, context, request=None, get_user_params=False, **kwargs):
        view = self.context
        catalog = get_utility(ICatalog)
        registry = get_current_registry()
        # check publication dates
        now = tztime(datetime.utcnow())
        params = Lt(catalog['effective_date'], now)
        # check workflow states
        wf_params = None
        for workflow in registry.getAllUtilitiesRegisteredFor(IWorkflow):
            wf_params = or_(wf_params, Any(catalog['workflow_state'], workflow.visible_states))
        params &= wf_params
        # check custom extensions
        get_all_params = True
        for name, adapter in sorted(registry.getAdapters((view,), IViewQueryParamsExtension),
                                    key=lambda x: x[1].weight):
            for new_params in adapter.get_params(context, request):
                if new_params is None:
                    return None
                elif new_params is END_PARAMS_MARKER:
                    get_all_params = False
                    break
                else:
                    params &= new_params
            else:
                continue
            break
        # activate search
        filters = Gt(catalog['push_end_date'], now)
        if get_all_params:
            # check content path
            content_path = view.get_content_path(context)
            if content_path is not None:
                filters &= Eq(catalog['parents'], content_path)
            # check content types
            if 'content_type' in kwargs:
                filters &= Eq(catalog['content_type'], kwargs['content_type'])
            else:
                filters &= NotAny(catalog['content_type'], IGNORED_CONTENT_TYPES.keys())
                content_types = view.get_content_types(context)
                if content_types:
                    filters &= Any(catalog['content_type'], content_types)
            # check data types
            data_types = view.get_data_types(context)
            if data_types:
                filters &= Any(catalog['data_type'], data_types)
            # check excluded content types
            content_types = view.get_excluded_content_types(context)
            if content_types:
                filters &= NotAny(catalog['content_type'], content_types)
            # check excluded data types
            data_types = view.get_excluded_data_types(context)
            if data_types:
                filters &= NotAny(catalog['data_type'], data_types)
            # check age limit
            age_limit = view.age_limit
            if age_limit:
                filters &= Gt(catalog['content_publication_date'],
                              now - timedelta(days=age_limit))
        params &= filters
        # add user search params
        if get_user_params:
            for name, adapter in registry.getAdapters((self,), IViewUserQuery):
                for user_param in adapter.get_user_params(request):
                    params &= user_param
        return params

    def get_results(self, context, sort_index, reverse, limit,
                    request=None, aggregates=None, settings=None, **kwargs):
        view = self.context
        catalog = get_utility(ICatalog)
        registry = get_current_registry()
        params = self.get_params(context, request, **kwargs)
        if params is None:
            items = CatalogResultSet([])
            total_count = 0
        else:
            if (not sort_index) or (sort_index == RELEVANCE_ORDER):
                sort_index = None
            query = CatalogQuery(catalog).query(params,
                                                sort_index=sort_index,
                                                reverse=reverse,
                                                limit=limit or 999)
            total_count = query[0]
            items = CatalogResultSet(query)
        for name, adapter in sorted(registry.getAdapters((view,), IViewQueryFilterExtension),
                                    key=lambda x: x[1].weight):
            items = adapter.filter(context, items, request)
        return unique_iter(items), total_count, {}


@subscriber(IObjectModifiedEvent, context_selector=IWfView)
def handle_modified_view(event):
    """Invalidate views cache when a view is modified"""
    view = event.object
    views_cache = get_cache(VIEWS_CACHE_REGION, VIEWS_CACHE_NAME)
    if view.is_using_context:
        views_cache.clear()
    else:
        intids = get_utility(IIntIds)
        cache_key = VIEW_CACHE_KEY.format(view=intids.queryId(view))
        views_cache.remove(cache_key)
