#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from pyramid.path import DottedNameResolver
from z3c.form import field
from z3c.form.interfaces import IDataExtractedEvent
from zope.interface import Interface, Invalid

from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION, MANAGE_TOOL_PERMISSION
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.common import IBaseSharedTool, IDashboardTypeColumnValue
from pyams_content.shared.common.interfaces.zmi import IDashboardTable
from pyams_content.shared.common.zmi.manager import SharedToolPropertiesEditForm
from pyams_content.shared.site import WfSiteTopic
from pyams_content.shared.site.interfaces import ISiteManager, ISiteManagerFactory
from pyams_content.shared.site.manager import SiteManager
from pyams_content.zmi.interfaces import ISiteTreeTable, IUserAddingsMenuLabel
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.container import ITableElementEditor, ITableElementName
from pyams_skin.interfaces.viewlet import IBreadcrumbItem, IMenuHeader, IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import DefaultElementEditorAdapter
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.registry import query_utility
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.interfaces.menu import ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@adapter_config(context=(ISiteManager, IPyAMSLayer, IDashboardTable),
                provides=IDashboardTypeColumnValue)
def site_manager_dashboard_type(context, request, view):
    """Site manager dashboard type column adapter"""
    return request.localizer.translate(context.content_name)


@adapter_config(context=(ISiteManager, ISiteManagementMenu), provides=IMenuHeader)
class SiteManagerSiteManagementMenuHeader(ContextRequestAdapter):
    """Site manager site management menu header adapter"""

    header = _("Site management")


@adapter_config(context=(ISiteManager, IAdminLayer), provides=IBreadcrumbItem)
class SiteManagerBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """Site manager breadcrumb adapter"""

    @property
    def label(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    css_class = 'strong'


@adapter_config(context=(ISiteManager, IPyAMSLayer), provides=ITableElementName)
class SiteManagerTableElementNameAdapter(ContextRequestAdapter):
    """Site manager table element name adapter"""

    @property
    def name(self):
        return II18n(self.context).query_attribute('short_name', request=self.request)


@adapter_config(context=(ISiteManager, IAdminLayer, Interface), provides=ITableElementEditor)
class SiteManagerTableElementEditorAdapter(DefaultElementEditorAdapter):
    """Site manager table element editor adapter"""

    view_name = 'admin#site-tree.html'
    modal_target = False


@adapter_config(context=(ISiteManager, IAdminLayer, ISiteTreeTable),
                provides=ITableElementEditor)
class SiteManagerTableElementEditor(DefaultElementEditorAdapter):
    """Site tree table element editor"""

    view_name = 'admin#site-tree.html'
    modal_target = False


@adapter_config(context=(ISiteManager, IAdminLayer), provides=IUserAddingsMenuLabel)
class SiteManagerUserAddingsMenuLabelAdapter(ContextRequestAdapter):
    """Site manager user addings menu label adapter"""

    @property
    def label(self):
        return '{content} ({site})'.format(
            content=self.request.localizer.translate(WfSiteTopic.content_name),
            site=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-site-manager.menu', context=ISiteRoot, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class SiteManagerAddMenu(ToolbarMenuItem):
    """Site manager add menu"""

    label = _("Add site manager")
    label_css_class = 'fa fa-fw fa-sitemap'
    url = 'add-site-manager.html'
    modal_target = True


@pagelet_config(name='add-site-manager.html', context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-site-manager.json', context=ISiteRoot, layer=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, base=AJAXAddForm)
class SiteManagerAddForm(AdminDialogAddForm):
    """Site manager add form"""

    title = _("Site manager")
    legend = _("Add site manager")
    icon_css_class = 'fa fa-fw fa-sitemap'

    fields = field.Fields(ISiteManager).select('title', 'short_name')
    edit_permission = None

    def create(self, data):
        factory = self.request.registry.settings.get('pyams_content.config.site_factory')
        if factory:
            factory = DottedNameResolver().resolve(factory)
        else:
            factory = ISiteManagerFactory(self.context, SiteManager)
        return factory()

    def add(self, object):
        short_name = II18n(object).query_attribute('short_name', request=self.request)
        name = translate_string(short_name, force_lower=True, spaces='-')
        self.context[name] = object

    def nextURL(self):
        return absolute_url(self.context, self.request, 'site-tree.html')


@pagelet_config(name='properties.html', context=ISiteManager, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json', context=ISiteManager, layer=IPyAMSLayer)
class SiteManagerPropertiesEditForm(SharedToolPropertiesEditForm):
    """Site manager properties edit form"""

    legend = _("Site folder properties")

    fields = field.Fields(ISiteManager).select('title', 'short_name', 'header', 'description',
                                               'notepad') + \
        field.Fields(IBaseSharedTool).select('shared_content_workflow') + \
        field.Fields(ISiteManager).select('navigation_mode', 'indexation_mode')


@subscriber(IDataExtractedEvent, form_selector=SiteManagerAddForm)
def handle_new_site_manager_data_extraction(event):
    """Handle new site manager data extraction"""
    container = event.form.context
    negotiator = query_utility(INegotiator)
    short_name = event.data.get('short_name', {}).get(negotiator.server_language)
    if not short_name:
        event.form.widgets.errors += (Invalid(_("You must provide a short name for default "
                                                "server language!")),)
        return
    name = translate_string(short_name, force_lower=True, spaces='-')
    if name in container:
        event.form.widgets.errors += (Invalid(_("Specified site manager name is already used!")),)
        return
    site = query_utility(ISiteManager, name=short_name)
    if site is not None:
        event.form.widgets.errors += (Invalid(_("A site manager is already registered with "
                                                "this name!!")),)
