#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyams_content import _

__docformat__ = 'restructuredtext'

from uuid import uuid4

from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import implementer
from zope.intid.interfaces import IIntIds
from zope.schema import Int

from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.site.interfaces import IContentLink, IExternalContentLink, \
    ISiteContainer, ISiteLink
from pyams_content.shared.site.link import ContentLink, ExternalContentLink
from pyams_content.shared.site.zmi.container import SiteContainerTreeTable
from pyams_content.shared.site.zmi.widget import SiteManagerFoldersSelectorFieldWidget
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.container import ITableElementName
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import get_object_name
from pyams_skin.viewlet.menu import MenuDivider
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_utility
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces import IPropertiesEditForm
from pyams_zmi.layer import IAdminLayer


@viewlet_config(name='add-link.divider', context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=CREATE_CONTENT_PERMISSION, weight=49)
class AddLinkMenuDivider(MenuDivider):
    """Add links menu divider"""


#
# Content link views
#

@viewlet_config(name='add-content-link.menu', context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=CREATE_CONTENT_PERMISSION, weight=50)
class ContentLinkAddMenu(ToolbarMenuItem):
    """Content link add menu"""

    label = _("Rent content...")
    label_css_class = 'fa fa-fw fa-external-link-square fa-rotate-90'
    url = 'add-content-link.html'
    modal_target = True


class IContentLinkAddFormFields(IContentLink):
    """Content link add forms fields interface"""

    parent = Int(title=_("Parent"),
                 description=_("Folder's parent"),
                 required=True)


@pagelet_config(name='add-content-link.html', context=ISiteContainer, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-content-link.json', context=ISiteContainer, layer=IPyAMSLayer,
             base=AJAXAddForm)
class ContentLinkAddForm(AdminDialogAddForm):
    """Content link add form"""

    legend = _("Rent existing content")

    fields = field.Fields(IContentLinkAddFormFields).select('reference', 'navigation_title',
                                                            'show_header', 'navigation_header',
                                                            'parent')
    fields['show_header'].widgetFactory = SingleCheckBoxFieldWidget
    fields['parent'].widgetFactory = SiteManagerFoldersSelectorFieldWidget

    edit_permission = CREATE_CONTENT_PERMISSION

    __target = None

    def updateWidgets(self, prefix=None):
        super(ContentLinkAddForm, self).updateWidgets(prefix)
        if 'parent' in self.widgets:
            self.widgets['parent'].permission = CREATE_CONTENT_PERMISSION
        if 'navigation_header' in self.widgets:
            self.widgets['navigation_header'].widget_css_class = 'input height-100'

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'reference', self.widgets,
                                         ('reference', 'navigation_title')))
        self.add_group(NamedWidgetsGroup(self, 'header', self.widgets,
                                         ('show_header', 'navigation_header'),
                                         fieldset_class='inner bordered',
                                         legend=_("Show navigation header"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=IContentLinkAddFormFields['show_header']))
        self.add_group(NamedWidgetsGroup(self, 'reference', self.widgets,
                                         ('parent',)))
        super(ContentLinkAddForm, self).updateGroups()

    def create(self, data):
        return ContentLink()

    def update_content(self, content, data):
        data = data.get(self, data)
        content.reference = data.get('reference')
        content.navigation_title = data['navigation_title']
        content.navigation_header = data['navigation_header']
        intids = get_utility(IIntIds)
        parent = intids.queryObject(data.get('parent'))
        if parent is not None:
            uuid = str(uuid4())
            parent[uuid] = content
            self.__target = parent

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.__target, self.request, 'admin#site-tree.html')

    def get_ajax_output(self, changes):
        return {'status': 'reload'}


@adapter_config(context=(IContentLink, IPyAMSLayer, ISiteTreeTable), provides=ITableElementName)
class ContentLinkTableElementName(ContextRequestViewAdapter):
    """Content link table element name"""

    @property
    def name(self):
        title = II18n(self.context).query_attribute('navigation_title', request=self.request)
        if not title:
            target = self.context.get_target()
            if target is not None:
                title = get_object_name(target, self.request, self.view)
        return '{title} <i class="fa fa-fw fa-external-link-square fa-rotate-90"></i>'.format(
            title=title or '--')


@pagelet_config(name='properties.html', context=IContentLink, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IContentLink, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class ContentLinkPropertiesEditForm(AdminDialogEditForm):
    """Content link properties edit form"""

    prefix = 'link_properties.'

    legend = _("Edit content link properties")

    fields = field.Fields(IContentLink).select('reference', 'navigation_title',
                                               'show_header', 'navigation_header')
    fields['show_header'].widgetFactory = SingleCheckBoxFieldWidget
    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(ContentLinkPropertiesEditForm, self).updateWidgets(prefix)
        if 'navigation_header' in self.widgets:
            self.widgets['navigation_header'].widget_css_class = 'input height-100'

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'reference', self.widgets,
                                         ('reference', 'navigation_title')))
        self.add_group(NamedWidgetsGroup(self, 'header', self.widgets,
                                         ('show_header', 'navigation_header'),
                                         fieldset_class='inner bordered',
                                         legend=_("Show navigation header"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=IContentLink['show_header']))
        super(ContentLinkPropertiesEditForm, self).updateGroups()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        intids = get_utility(IIntIds)
        if changes:
            table = SiteContainerTreeTable(self.context.__parent__, self.request)
            table.update()
            row = table.setUpRow(self.context)
            output.setdefault('events', []).append({
                'event': 'myams.refresh',
                'options': {
                    'handler': 'MyAMS.skin.refreshRow',
                    'object_id': '{0}::{1}'.format(SiteContainerTreeTable.id,
                                                   intids.queryId(self.context)),
                    'row': table.renderRow(row)
                }
            })
        return output


#
# External content link views
#

@viewlet_config(name='add-external-link.menu', context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=CREATE_CONTENT_PERMISSION, weight=60)
class ExternalContentLinkAddMenu(ToolbarMenuItem):
    """External content link add menu"""

    label = _("External content link...")
    label_css_class = 'fa fa-fw fa-link fa-rotate-90'
    url = 'add-external-link.html'
    modal_target = True


class IExternalContentLinkAddFormFields(IExternalContentLink):
    """External content link add forms fields interface"""

    parent = Int(title=_("Parent"),
                 description=_("Folder's parent"),
                 required=True)


@pagelet_config(name='add-external-link.html', context=ISiteContainer, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-external-link.json', context=ISiteContainer, layer=IPyAMSLayer,
             base=AJAXAddForm)
class ExternalContentLinkAddForm(AdminDialogAddForm):
    """External content link add form"""

    legend = _("Link external content")

    fields = field.Fields(IExternalContentLinkAddFormFields).select('url', 'navigation_title',
                                                                    'show_header',
                                                                    'navigation_header', 'parent')
    fields['parent'].widgetFactory = SiteManagerFoldersSelectorFieldWidget

    edit_permission = CREATE_CONTENT_PERMISSION

    __target = None

    def updateWidgets(self, prefix=None):
        super(ExternalContentLinkAddForm, self).updateWidgets(prefix)
        if 'parent' in self.widgets:
            self.widgets['parent'].permission = CREATE_CONTENT_PERMISSION
        if 'navigation_header' in self.widgets:
            self.widgets['navigation_header'].widget_css_class = 'input height-100'

    def create(self, data):
        return ExternalContentLink()

    def update_content(self, content, data):
        data = data.get(self, data)
        content.url = data.get('url')
        content.navigation_title = data['navigation_title']
        content.navigation_header = data['navigation_header']
        intids = get_utility(IIntIds)
        parent = intids.queryObject(data.get('parent'))
        if parent is not None:
            uuid = str(uuid4())
            parent[uuid] = content
            self.__target = parent

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.__target, self.request, 'admin#site-tree.html')

    def get_ajax_output(self, changes):
        return {'status': 'reload'}


@adapter_config(context=(IExternalContentLink, IPyAMSLayer, ISiteTreeTable),
                provides=ITableElementName)
class ExternalContentLinkTableElementName(ContextRequestViewAdapter):
    """External content link table element name"""

    @property
    def name(self):
        title = II18n(self.context).query_attribute('navigation_title', request=self.request)
        if not title:
            title = self.context.url
        return '{title} <i class="fa fa-fw fa-link fa-rotate-90"></i>'.format(
            title=title or '--')


@pagelet_config(name='properties.html', context=IExternalContentLink, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IExternalContentLink, layer=IPyAMSLayer)
@implementer(IPropertiesEditForm)
class ExternalContentLinkPropertiesEditForm(AdminDialogEditForm):
    """External content link properties edit form"""

    prefix = 'link_properties.'

    legend = _("Edit external content link properties")
    dialog_class = 'modal-large'

    fields = field.Fields(IExternalContentLink).select('url', 'navigation_title',
                                                       'navigation_header')
    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(ExternalContentLinkPropertiesEditForm, self).updateWidgets(prefix)
        if 'navigation_header' in self.widgets:
            self.widgets['navigation_header'].widget_css_class = 'input height-100'

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        intids = get_utility(IIntIds)
        if changes:
            table = SiteContainerTreeTable(self.context.__parent__, self.request)
            table.update()
            row = table.setUpRow(self.context)
            output.setdefault('events', []).append({
                'event': 'myams.refresh',
                'options': {
                    'handler': 'MyAMS.skin.refreshRow',
                    'object_id': '{0}::{1}'.format(SiteContainerTreeTable.id,
                                                   intids.queryId(self.context)),
                    'row': table.renderRow(row)
                }
            })
        return output
