#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json
from datetime import datetime

from pyramid.decorator import reify
from pyramid.exceptions import NotFound
from pyramid.location import lineage
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, alsoProvides, implementer
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent import ObjectMovedEvent

from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, MANAGE_SITE_PERMISSION
from pyams_content.profile.interfaces import IAdminProfile
from pyams_content.shared.common.interfaces import ISharedContent, ISharedSite
from pyams_content.shared.common.interfaces.zmi import IDashboardTable
from pyams_content.shared.common.zmi.dashboard import SharedToolDashboardOwnerColumn, \
    SharedToolDashboardSequenceColumn, SharedToolDashboardStatusColumn, \
    SharedToolDashboardStatusDateColumn, SharedToolDashboardStatusPrincipalColumn, \
    SharedToolDashboardVersionColumn
from pyams_content.shared.site import WfSiteTopic
from pyams_content.shared.site.interfaces import IBaseSiteItem, IContentLink, ISiteContainer, \
    ISiteLink, ISiteManager
from pyams_content.zmi import pyams_content
from pyams_content.zmi.interfaces import ISiteTreeMenu, ISiteTreeTable, IUserAddingsMenuLabel
from pyams_form.form import ajax_config
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.interfaces import ISequentialIdInfo, ISequentialIdTarget, ISequentialIntIds
from pyams_skin.container import ContainerView, delete_container_element
from pyams_skin.event import get_json_table_cell_refresh_event
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor, ITableElementName, \
    ITableWithActions
from pyams_skin.interfaces.viewlet import IBreadcrumbItem, ITableItemColumnActionsMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.skin import apply_skin
from pyams_skin.table import BaseTable, DefaultElementEditorAdapter, I18nColumn, JsActionColumn, \
    NameColumn, SorterColumn, TrashColumn
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, NullAdapter, \
    adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.interfaces.traversing import IPathElements
from pyams_utils.registry import get_utility
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import contentprovider_config, viewlet_config
from pyams_workflow.interfaces import IWorkflowPublicationInfo, IWorkflowVersions
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu, ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


@contentprovider_config(name='content_header', context=ISharedSite,
                        view=Interface, layer=IPyAMSLayer)
@contentprovider_config(name='content_header', context=ISiteContainer,
                        view=Interface, layer=IPyAMSLayer)
@template_config(template='templates/container-header.pt', layer=IPyAMSLayer)
class SiteContainerHeaderContentProvider(object):
    """Site container header content provider"""

    back_url = '#site-tree.html'
    back_target = None

    icon_class = ''

    def __init__(self, context, request, view):
        super(SiteContainerHeaderContentProvider, self).__init__(context, request, view)
        # check publication state
        translate = self.request.localizer.translate
        now = tztime(datetime.utcnow())
        pub_info = IWorkflowPublicationInfo(self.context)
        if pub_info.publication_effective_date:
            if pub_info.publication_effective_date <= now:
                if pub_info.publication_expiration_date:
                    if pub_info.publication_effective_date > now:
                        state = _("Retired")
                        state_label = _("{state} since {from_date}")
                    else:
                        state = _("Published")
                        state_label = _("{state} since {from_date} until {to_date}")
                else:
                    state = _("Published")
                    state_label = _("{state} since {from_date}")
            else:
                if pub_info.publication_expiration_date:
                    state = _("To be published")
                    state_label = _("{state} from {from_date} to {to_date}")
                else:
                    state = _("Published")
                    state_label = _("{state} after {to_date}")
            self.state = translate(state_label).format(
                state='<span class="text-danger">{}</span>'.format(translate(state)),
                from_date=format_datetime(pub_info.publication_effective_date),
                to_date=format_datetime(pub_info.publication_expiration_date))
        else:
            self.state = '<span class="text-danger">{}</span>'.format(
                translate(_("Not published")))

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    @reify
    def hex_oid(self):
        return ISequentialIdInfo(self.context).hex_oid

    @property
    def favorite_content(self):
        profile = IAdminProfile(self.request.principal)
        return self.hex_oid in (profile.favorites or '')


@adapter_config(context=(IBaseSiteItem, IAdminLayer), provides=IBreadcrumbItem)
class SiteContainerBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """Site container breadcrumb adapter"""

    @property
    def label(self):
        return II18n(self.context).query_attribute('title', request=self.request)


@adapter_config(context=(ISiteContainer, IAdminLayer), provides=IUserAddingsMenuLabel)
class SiteManagerUserAddingsMenuLabelAdapter(ContextRequestAdapter):
    """Site container user adding menu label adapter"""

    @property
    def label(self):
        return '{content} ({site})'.format(
            content=self.request.localizer.translate(WfSiteTopic.content_name),
            site=II18n(self.context).query_attribute('title', request=self.request))


#
# Site container publication views
#

@viewlet_config(name='workflow-publication.menu', context=IBaseSiteItem, layer=IPyAMSLayer,
                view=ISiteTreeTable, manager=ITableItemColumnActionsMenu,
                permission=MANAGE_SITE_PERMISSION, weight=810)
class SiteContainerTableItemWorkflowPublicationMenu(ToolbarMenuItem):
    """Site container tree item workflow publication menu"""

    label = _("Publication dates...")
    label_css_class = 'fa fa-fw fa-eye'
    url = 'workflow-publication.html'
    modal_target = True
    stop_propagation = True


@viewlet_config(name='workflow-publication.menu', context=IBaseSiteItem, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_SITE_PERMISSION, weight=510)
class SiteContainerWorkflowPublicationMenu(MenuItem):
    """Site container workflow publication menu"""

    label = _("Publication dates...")
    icon_class = 'fa-eye'
    url = 'workflow-publication.html'
    modal_target = True


@pagelet_config(name='workflow-publication.html', context=IBaseSiteItem, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='workflow-publication.json', context=IBaseSiteItem, layer=IPyAMSLayer)
class SiteContainerWorkflowPublicationEditForm(AdminDialogEditForm):
    """Site container workflow publication edit form"""

    prefix = 'site_publication.'

    legend = _("Update publication dates")

    fields = field.Fields(IWorkflowPublicationInfo).select('publication_effective_date',
                                                           'publication_expiration_date')
    edit_permission = MANAGE_SITE_PERMISSION

    def updateActions(self):
        super(SiteContainerWorkflowPublicationEditForm, self).updateActions()
        submit = self.actions.get('submit')
        if submit is not None:
            submit.object_data = {'ams-focus-target': 'true'}
            alsoProvides(submit, IObjectData)

    def updateWidgets(self, prefix=None):
        super(SiteContainerWorkflowPublicationEditForm, self).updateWidgets(prefix)
        if 'publication_effective_date' in self.widgets:
            pub_info = IWorkflowPublicationInfo(self.context)
            if pub_info.publication_effective_date is None:
                self.widgets['publication_effective_date'].value = \
                    tztime(datetime.utcnow()).strftime('%d/%m/%y %H:%M')

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if changes:
            info = IWorkflowPublicationInfo(self.context, None)
            if info is not None:
                if info.is_published():
                    icon_class = 'fa-eye opacity-75'
                else:
                    icon_class = 'fa-eye-slash text-danger opaque'
                value = '<i class="fa fa-fw {icon_class} hint align-base" title="{title}" ' \
                        'data-ams-hint-gravity="e"></i>'.format(
                    icon_class=icon_class,
                    title=self.request.localizer.translate(_("Visible element?")))
                intids = get_utility(IIntIds)
                output.setdefault('events', []).append(
                    get_json_table_cell_refresh_event(self.context, self.request,
                                                      '{0}::{1}'.format(SiteContainerTreeTable.id,
                                                                        intids.queryId(self.context)),
                                                      'visible',
                                                      value))
        return output


#
# Site container tree view
#

@viewlet_config(name='site-tree.menu', layer=IAdminLayer, context=IBaseSiteItem,
                manager=ISiteManagementMenu, permission=VIEW_SYSTEM_PERMISSION, weight=10)
@viewletmanager_config(name='site-tree.menu', layer=IAdminLayer, context=IBaseSiteItem,
                       provides=ISiteTreeMenu)
@implementer(ISiteTreeMenu)
class SiteContainerTreeMenu(MenuItem):
    """Site container tree menu"""

    label = _("Site tree")
    icon_class = 'fa-sitemap'
    url = '#site-tree.html'


@implementer(IDashboardTable, ISiteTreeTable, ITableWithActions)
class SiteContainerTreeTable(BaseTable):
    """Site container tree table"""

    id = 'site_tree_table'
    title = _("Site tree")

    hide_body_toolbar = True
    sortOn = None

    permission = MANAGE_SITE_PERMISSION

    def __init__(self, context, request, can_sort=False, rows_state=None):
        super(SiteContainerTreeTable, self).__init__(context, request)
        self.can_sort = can_sort
        self.rows_state = rows_state

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if self.can_sort and ((not permission) or self.request.has_permission(permission,
                                                                              context=self.context)):
            classes.append('table-dnd')
        return {
            'table': ' '.join(classes),
            'tr.selected': lambda item, col, row: 'current' if item is self.context else ''
        }

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        intids = get_utility(IIntIds)
        manager = get_parent(self.context, ISiteManager)
        attributes.setdefault('table', {}).update({
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-datatable-sort': 'false',
            'data-ams-datatable-pagination': 'false',
            'data-ams-tree-node-id': intids.queryId(manager),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop': 'MyAMS.tree.sortTree',
            'data-ams-tablednd-drop-target': 'set-site-order.json'
        })
        attributes.setdefault('tr', {}).update({
            'id': lambda x, col: '{0}::{1}'.format(self.id, intids.queryId(x)),
            'data-ams-location': lambda x, col: absolute_url(x.__parent__, self.request),
            'data-ams-tree-node-id': lambda x, col: intids.queryId(x),
            'data-ams-tree-node-parent-id': lambda x, col: intids.queryId(x.__parent__)
        })
        return attributes


@adapter_config(name='sorter',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeSorterColumn(SorterColumn):
    """Site container tree sorter column"""

    permission = MANAGE_SITE_PERMISSION

    def renderCell(self, item):
        if self.table.can_sort:
            return super(SiteContainerTreeSorterColumn, self).renderCell(item)
        else:
            return ''


@view_config(name='set-site-order.json', context=IBaseSiteItem, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_PERMISSION, renderer='json', xhr=True)
def set_site_order(request):
    """Set site elements order"""
    intids = get_utility(IIntIds)
    parent_oid = int(request.params.get('parent'))
    new_parent = intids.queryObject(parent_oid)
    # check for changing parent
    if request.params.get('action') == 'reparent':
        child_oid = int(request.params.get('child'))
        child = intids.queryObject(child_oid)
        # check if new parent is not a previous child
        parent_path = IPathElements(new_parent)
        if child_oid in parent_path.parents:
            return {
                'status': 'reload',
                'smallbox': {
                    'status': 'error',
                    'message': request.localizer.translate(_("Can't reparent object to one of it's children. "
                                                             "Reloading...")),
                    'timeout': 5000
                }
            }
        old_parent = child.__parent__
        new_name = old_name = child.__name__
        if old_name in new_parent:
            index = 1
            new_name = '{name}-{index:02}'.format(name=old_name, index=index)
            while new_name in new_parent:
                index += 1
                new_name = '{name}-{index:02}'.format(name=old_name, index=index)
        new_parent[new_name] = child
        del old_parent[old_name]
        request.registry.notify(ObjectMovedEvent(child, old_parent, old_name, new_parent, new_name))
    # Re-define order
    if len(new_parent.keys()) > 1:
        names = [child.__name__ for child in [intids.queryObject(oid)
                                              for oid in map(int, json.loads(request.params.get('order')))]
                 if (child is not None) and (child.__parent__ is new_parent)]
        if names:
            new_parent.updateOrder(names)
    # get all new parent child
    table = SiteContainerTreeTable(request.context, request,
                                   can_sort=json.loads(request.params.get('can_sort', 'false')),
                                   rows_state='plus')
    table.update()
    result = []
    for item in new_parent.values():
        row = table.setUpRow(item)
        result.append(table.renderRow(row).strip())
    return result


@adapter_config(name='visible',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeVisibleColumn(JsActionColumn):
    """Site container tree visible column"""

    cssClasses = {'th': 'action',
                  'td': 'action'}

    icon_class = 'fa fa-fw fa-eye'
    inactive_icon_hint = _("Visible element?")
    active_icon_hint = _("Switch element visibility")

    url = 'PyAMS_content.site.switchVisibility'
    permission = MANAGE_CONTENT_PERMISSION

    weight = 5

    def get_icon(self, item):
        if ISiteLink.providedBy(item):
            icon_class = 'fa-eye' if item.visible else 'fa-eye-slash'
            if not IWorkflowPublicationInfo(item.__parent__).is_published():
                icon_class += ' text-danger'
            return '<i class="fa fa-fw {icon_class}"></i>'.format(icon_class=icon_class)
        if ISharedContent.providedBy(item):
            item = IWorkflowVersions(item).get_last_versions(count=1)[-1]
        info = IWorkflowPublicationInfo(item, None)
        if info is None:
            return ''
        if info.is_published():
            icon_class = 'fa-eye opacity-75'
        else:
            icon_class = 'fa-eye-slash text-danger opaque'
        return '<i class="fa fa-fw {icon_class} hint align-base" title="{title}" ' \
               'data-ams-hint-gravity="e"></i>'.format(
            icon_class=icon_class,
            title=self.request.localizer.translate(self.get_icon_hint(item)))

    def get_icon_hint(self, item):
        translate = self.request.localizer.translate
        if ISiteLink.providedBy(item) and self.request.has_permission(self.permission,
                                                                      context=item):
            return translate(self.active_icon_hint)
        return translate(self.inactive_icon_hint)

    def renderCell(self, item):
        if ISiteLink.providedBy(item) and self.request.has_permission(self.permission,
                                                                      context=item):
            return super().renderCell(item)
        return self.get_icon(item)


@view_config(name='switch-content-visibility.json', context=IBaseSiteItem,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def switch_content_visibility(request):
    """Switch content link visibility"""
    container = ISiteContainer(request.context)
    content = container.get(str(request.params.get('object_name')))
    if not ISiteLink.providedBy(content):
        raise NotFound()
    content.visible = not content.visible
    return {
        'visible': content.visible,
        'published': IWorkflowPublicationInfo(content.__parent__).is_published()
    }


@adapter_config(name='type',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeTypeColumn(NullAdapter):
    """Site container tree type column -- disabled in site tree view!"""


@adapter_config(name='name',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeNameColumn(NameColumn):
    """Site container tree name column"""

    _header = _("Folders and topics")

    def renderHeadCell(self):
        return '<span data-ams-stop-propagation="true"' \
               '      data-ams-click-handler="MyAMS.tree.switchTree">' \
               '    <span class="small hint" title="{hint}" data-ams-hint-gravity="e">' \
               '        <i class="fa fa-fw fa-plus-square-o switch"></i>' \
               '    </span>&nbsp;&nbsp;{title}' \
               '</span>'.format(
            hint=self.request.localizer.translate(_("Click to open/close all folders")),
            title=super().renderHeadCell())

    def renderCell(self, item, name=None):
        depth = -3
        for node in lineage(item):
            depth += 1
        return '<div class="name">' \
               '    {padding}' \
               '    <span class="small hint" title="{hint}" data-ams-hint-gravity="e" ' \
               '          data-ams-click-handler="MyAMS.tree.switchTableNode"' \
               '          data-ams-stop-propagation="true">' \
               '        <i class="fa fa-fw {switch}"></i>' \
               '    </span>&nbsp;&nbsp;<span class="title">{title}</span>' \
               '</div>'.format(
                padding='<span class="tree-node-padding"></span>' * depth,
                hint=self.request.localizer.translate(_("Click to show/hide inner folders")),
                switch='fa-{state}-square-o switch'.format(
                    state=self.table.rows_state or ('minus' if item in lineage(self.context) else 'plus'))
                        if ISiteContainer.providedBy(item) else '',
                title=name or super().renderCell(item))


@adapter_config(name='content-type',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerContentTypeColumn(I18nColumn, GetAttrColumn):
    """Site container content type column"""

    _header = _("Content")

    weight = 12

    def getValue(self, obj):
        return self.request.localizer.translate(obj.content_name)


@adapter_config(name='sequence',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeSequenceColumn(SharedToolDashboardSequenceColumn):
    """Site container tree OID column"""

    def getValue(self, obj):
        target = obj.get_target() if IContentLink.providedBy(obj) else obj
        if IContentLink.providedBy(obj):
            sequence = None
            sequence_prefix = ''
            if target is not None:
                sequence_target = get_parent(target, ISequentialIdTarget)
                if sequence_target is not None:
                    sequence = get_utility(ISequentialIntIds,
                                           name=sequence_target.sequence_name)
                    sequence_prefix = sequence_target.sequence_prefix
            if sequence is None:
                sequence = get_utility(ISequentialIntIds, name='')
            oid = sequence.get_internal_id(obj.reference)
            return '&raquo; {0}'.format(sequence.get_base_oid(oid, sequence_prefix))
        target = obj
        if ISharedContent.providedBy(target):
            target = IWorkflowVersions(target).get_last_versions()[-1]
        return super().getValue(target)


@adapter_config(name='status',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeStatusColumn(SharedToolDashboardStatusColumn):
    """Site container tree status column"""

    def getValue(self, obj):
        target = obj.get_target() if IContentLink.providedBy(obj) else obj
        if ISharedContent.providedBy(target):
            target = IWorkflowVersions(target).get_last_versions()[-1]
        result = super().getValue(target)
        if IContentLink.providedBy(obj):
            result = '({0})'.format(result)
        return result


@adapter_config(name='status_date',
                context=(IBaseSiteItem, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SiteContainerTreeStatusDateColumn(SharedToolDashboardStatusDateColumn):
    """Site container tree status date column"""

    def getValue(self, obj):
        target = obj.get_target() if IContentLink.providedBy(obj) else obj
        if ISharedContent.providedBy(target):
            target = IWorkflowVersions(target).get_last_versions()[-1]
        result = super().getValue(target)
        if IContentLink.providedBy(obj):
            result = '({0})'.format(result)
        return result


@adapter_config(name='version',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeVersionColumn(SharedToolDashboardVersionColumn):
    """Site container tree version column"""

    def getValue(self, obj):
        target = obj.get_target() if IContentLink.providedBy(obj) else obj
        if ISharedContent.providedBy(target):
            target = IWorkflowVersions(target).get_last_versions()[-1]
        result = super().getValue(target)
        if IContentLink.providedBy(obj):
            result = '({0})'.format(result)
        return result


@adapter_config(name='status_principal',
                context=(IBaseSiteItem, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SiteContainerTreeStatusPrincipalColumn(SharedToolDashboardStatusPrincipalColumn):
    """Site container tree status principal column"""

    def getValue(self, obj):
        target = obj.get_target() if IContentLink.providedBy(obj) else obj
        if ISharedContent.providedBy(target):
            target = IWorkflowVersions(target).get_last_versions()[-1]
        result = super().getValue(target)
        if IContentLink.providedBy(obj):
            result = '({0})'.format(result)
        return result


@adapter_config(name='owner',
                context=(IBaseSiteItem, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SiteContainerTreeOwnerColumn(SharedToolDashboardOwnerColumn):
    """Site container tree owner column"""

    def getValue(self, obj):
        target = obj.get_target() if IContentLink.providedBy(obj) else obj
        if ISharedContent.providedBy(target):
            target = IWorkflowVersions(target).get_last_versions()[-1]
        result = super().getValue(target)
        if IContentLink.providedBy(obj):
            result = '({0})'.format(result)
        return result


@adapter_config(name='trash',
                context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IColumn)
class SiteContainerTreeTrashColumn(TrashColumn):
    """Site container tree trash column"""

    icon_hint = _("Delete site item")
    permission = MANAGE_SITE_PERMISSION

    def has_permission(self, item):
        if item in lineage(self.context):
            return False
        return super().has_permission(item) and item.is_deletable()


@view_config(name='delete-element.json', context=ISiteContainer, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_PERMISSION, renderer='json', xhr=True)
def delete_site_item(request):
    """Delete item from site container"""
    return delete_container_element(request, ignore_permission=True)


@adapter_config(context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=IValues)
class SiteContainerTreeValuesAdapter(ContextRequestViewAdapter):
    """Site container tree values adapter"""

    @property
    def values(self):

        def get_values(container, result):
            if container not in result:
                result.append(container)
            if ISiteContainer.providedBy(container) and (container in lineage(self.context)):
                for child in container.values():
                    get_values(child, result)
            return result

        manager = get_parent(self.context, ISiteManager)
        values = []
        for container in manager.values():
            values.append(container)
            if ISiteContainer.providedBy(container):
                get_values(container, values)
        return values


@pagelet_config(name='site-tree.html', context=IBaseSiteItem, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteContainerTreeView(AdminView, ContainerView):
    """Site Container tree view"""

    table_class = SiteContainerTreeTable

    def __init__(self, context, request):
        super(ContainerView, self).__init__(context, request)
        self.table = SiteContainerTreeTable(context, request,
                                            can_sort=ISiteManager.providedBy(context))


@adapter_config(context=(IBaseSiteItem, IAdminLayer, ISiteTreeTable),
                provides=IPageHeader)
class SiteContainerViewHeaderAdapter(DefaultPageHeaderAdapter):
    """Site container tree view header adapter"""

    icon_class = 'fa fa-fw fa-sitemap'


@view_config(name='get-tree.json', context=IBaseSiteItem, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def get_tree(request):
    """Get whole tree"""

    def get_tree_values(parent):
        """Iterator over container tree items"""
        for item in parent.values():
            yield item
            if ISiteContainer.providedBy(item):
                yield from get_tree_values(item)

    apply_skin(request, 'PyAMS admin skin')
    table = SiteContainerTreeTable(request.context, request,
                                   can_sort=json.loads(request.params.get('can_sort', 'false')))
    table.update()
    result = []
    manager = get_parent(request.context, ISiteManager)
    for item in get_tree_values(manager):
        row = table.setUpRow(item)
        result.append(table.renderRow(row).strip())
    return result


@view_config(name='get-tree-nodes.json', context=ISiteContainer, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def get_tree_nodes(request):
    """Get tree nodes"""
    apply_skin(request, 'PyAMS admin skin')
    table = SiteContainerTreeTable(request.context, request,
                                   can_sort=json.loads(request.params.get('can_sort', 'false')))
    table.update()
    result = []
    for item in request.context.values():
        row = table.setUpRow(item)
        result.append(table.renderRow(row).strip())
    return result


@adapter_config(context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=ITableElementName)
class SiteContainerTableElementName(ContextRequestViewAdapter):
    """Site container tree table element name"""

    @property
    def name(self):
        return II18n(self.context).query_attribute('title', request=self.request)


@adapter_config(context=(IBaseSiteItem, IPyAMSLayer, ISiteTreeTable),
                provides=ITableElementEditor)
class SiteContainerTableElementEditor(DefaultElementEditorAdapter):
    """Site container tree table element editor"""

    view_name = 'admin#site-tree.html'
    modal_target = False


@adapter_config(context=(ISharedContent, IPyAMSLayer, ISiteTreeTable),
                provides=ITableElementName)
class SharedContentTableElementName(ContextRequestViewAdapter):
    """Shared content tree table element name"""

    @property
    def name(self):
        version = IWorkflowVersions(self.context).get_last_versions(count=1)[0]
        return II18n(version).query_attribute('title', request=self.request)


@adapter_config(context=(ISharedContent, IPyAMSLayer, ISiteTreeTable),
                provides=ITableElementEditor)
class SharedContentTableElementEditor(DefaultElementEditorAdapter):
    """Shared content tree table element editor"""

    view_name = 'admin'
    modal_target = False

    @property
    def url(self):
        version = IWorkflowVersions(self.context).get_last_versions(count=1)[0]
        return absolute_url(version, self.request, self.view_name)
