#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.intid.interfaces import IIntIds
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content import _
from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.component.links import InternalLink
from pyams_content.features.menu import IDynamicMenu
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.interfaces import MANAGE_SITE_PERMISSION
from pyams_content.shared.common.manager import BaseSharedTool
from pyams_content.shared.site.container import SiteContainerMixin
from pyams_content.shared.site.interfaces import ISiteFolder, ISiteFolderRoles, ISiteManager
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext
from pyams_security.interfaces import IDefaultProtectionPolicy
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.container import find_objects_providing
from pyams_utils.registry import get_local_registry
from pyams_utils.request import query_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@implementer(IDefaultProtectionPolicy, ISiteFolder, ISiteFolderRoles,
             IIllustrationTarget, ILinkIllustrationTarget, IPortalContext, IPreviewTarget)
class SiteFolder(SiteContainerMixin, OrderedContainer, BaseSharedTool):
    """Site folder persistent class"""

    roles_interface = ISiteFolderRoles

    header = FieldProperty(ISiteFolder['header'])
    description = FieldProperty(ISiteFolder['description'])
    notepad = FieldProperty(ISiteFolder['notepad'])

    visible_in_list = FieldProperty(ISiteFolder['visible_in_list'])
    navigation_title = FieldProperty(ISiteFolder['navigation_title'])
    navigation_mode = FieldProperty(ISiteFolder['navigation_mode'])

    content_name = _("Site folder")

    sequence_name = ''  # use default sequence generator
    sequence_prefix = ''

    def is_deletable(self):
        for element in self.values():
            if not element.is_deletable():
                return False
        return True


@adapter_config(context=ISiteFolder, provides=IFormContextPermissionChecker)
class SiteFolderPermissionChecker(ContextAdapter):
    """Site folder edit permission checker"""

    edit_permission = MANAGE_SITE_PERMISSION


@adapter_config(context=ISiteFolder, provides=IDynamicMenu)
def site_folder_dynamic_menu_factory(context):
    """Site folder dynamic menu factory"""
    result = InternalLink()
    result.title = context.navigation_title.copy() if context.navigation_title else {}
    result.reference = ISequentialIdInfo(context).hex_oid
    return result


@vocabulary_config(name='PyAMS site folders')
class SiteManagerFoldersVocabulary(SimpleVocabulary):
    """Site manager folders vocabulary"""

    def __init__(self, context):
        terms = []
        site = get_parent(context, ISiteManager)
        if site is not None:
            registry = get_local_registry()
            if registry is not None:
                request = query_request()
                intids = registry.getUtility(IIntIds)
                for folder in find_objects_providing(site, ISiteFolder):
                    terms.append(SimpleTerm(value=intids.queryId(folder),
                                            title=II18n(folder).query_attribute('title',
                                                                                request=request)))
        super(SiteManagerFoldersVocabulary, self).__init__(terms)
