#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, \
    BaseParagraphFactory, IParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker, \
    MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json import IJSONConverter
from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_content.shared.logo.interfaces import ILogosParagraph, LOGOS_PARAGRAPH_NAME, \
    LOGOS_PARAGRAPH_RENDERERS, LOGOS_PARAGRAPH_TYPE
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_sequence.interfaces import IInternalReferencesList
from pyams_sequence.reference import get_reference_target, get_visible_version
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.url import canonical_url
from pyams_utils.vocabulary import vocabulary_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowState


__docformat__ = 'restructuredtext'

from pyams_content import _


@factory_config(ILogosParagraph)
@implementer(IInternalReferencesList)
class LogosParagraph(BaseParagraph):
    """Logos paragraph"""

    factory_name = LOGOS_PARAGRAPH_TYPE
    icon_class = 'fa-th-large'
    icon_hint = LOGOS_PARAGRAPH_NAME

    logos = FieldProperty(ILogosParagraph['logos'])
    renderer = FieldProperty(ILogosParagraph['renderer'])

    def get_logos(self, state=None, with_reference=False, getter=None):
        for reference in self.logos or ():
            target = get_reference_target(reference, state, getter=getter)
            if target is not None:
                yield (reference, target) if with_reference else target

    # Internal references interface support

    @property
    def references(self):
        return self.logos

    def get_targets(self, state=None):
        yield from self.get_logos(state)


@utility_config(name=LOGOS_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class LogosParagraphFactory(BaseParagraphFactory):
    """Logos paragraph factory"""

    name = LOGOS_PARAGRAPH_NAME
    content_type = LogosParagraph
    secondary_menu = True


@adapter_config(required=ILogosParagraph,
                provides=IContentChecker)
class LogosParagraphContentChecker(BaseParagraphContentChecker):
    """Logos paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(ILogosParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        field_title = translate(ILogosParagraph['logos'].title)
        if not self.context.logos:
            output.append(translate(ERROR_VALUE).format(
                field=field_title,
                message=translate(_("no selected logo"))))
        else:
            for reference, target in self.context.get_logos(with_reference=True):
                if target is None:
                    output.append(translate(ERROR_VALUE).format(
                        field=field_title,
                        message=translate(_("logo '{0}' can't be found")).format(reference)))
                else:
                    workflow = IWorkflow(target, None)
                    if workflow is not None:
                        workflow_state = IWorkflowState(target)
                        if workflow_state.state not in workflow.published_states:
                            output.append(translate(ERROR_VALUE).format(
                                field=field_title,
                                message=translate(_("logo '{0}' is not published")).format(
                                    II18n(target).query_attribute('title', request=request))))
        return output


@vocabulary_config(name=LOGOS_PARAGRAPH_RENDERERS)
class LogosParagraphRendererVocabulary(SimpleVocabulary):
    """Logos paragraph renderers vocabulary"""

    def __init__(self, context=None):
        request = check_request()
        translate = request.localizer.translate
        registry = request.registry
        if not ILogosParagraph.providedBy(context):
            context = LogosParagraph()
        terms = [SimpleTerm(name, title=translate(adapter.label))
                 for name, adapter in sorted(registry.getAdapters((context, request), IContentRenderer),
                                             key=lambda x: x[1].weight)]
        super().__init__(terms)


@adapter_config(required=(ILogosParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONLogosParagraphConverter(JSONParagraphConverter):
    """JSON logos paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        logos = list(self.context.get_logos(getter=get_visible_version))
        if logos:
            result['items'] = [
                {
                    'title': logo.get_title(self.request),
                    'image': self.get_image_url(logo, 'image', params),
                    'href': canonical_url(logo, self.request)
                }
                for logo in logos
            ]
        return result
