#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import implementer, provider
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.links import InternalReferenceMixin
from pyams_content.component.theme.interfaces import IThemesTarget
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_VALUE
from pyams_content.features.review.interfaces import IReviewTarget
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, \
    WfSharedContentChecker, register_content_type
from pyams_content.shared.logo.interfaces import ILogo, IWfLogo, IWfLogoFactory, \
    LOGO_CONTENT_NAME, LOGO_CONTENT_TYPE
from pyams_file.property import FileProperty
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.url import ICanonicalURL, IRelativeURL
from pyams_utils.url import canonical_url, relative_url


__docformat__ = 'restructuredtext'

from pyams_content import _


@implementer(IWfLogo, IThemesTarget, IReviewTarget)
class WfLogo(WfSharedContent, InternalReferenceMixin):
    """Logo persistent class"""

    content_type = LOGO_CONTENT_TYPE
    content_name = LOGO_CONTENT_NAME

    handle_content_url = False
    handle_header = False
    handle_description = False

    acronym = FieldProperty(IWfLogo['acronym'])
    url = FieldProperty(IWfLogo['url'])
    reference = FieldProperty(IWfLogo['reference'])
    alt_title = FieldProperty(IWfLogo['alt_title'])
    image = FileProperty(IWfLogo['image'])
    monochrome_image = FileProperty(IWfLogo['monochrome_image'])

    def get_title(self, request):
        i18n = II18n(self)
        return i18n.query_attribute('alt_title', request=request) or \
            i18n.query_attribute('title', request=request)


register_content_type(WfLogo, shared_content=False)


@provider(IWfLogoFactory)
@implementer(ILogo)
class Logo(SharedContent):
    """Workflow managed logo persistent class"""


@adapter_config(context=IWfLogoFactory, provides=IWfSharedContentFactory)
def logo_content_factory(context):
    return WfLogo


@adapter_config(name='properties', context=IWfLogo, provides=IContentChecker)
class WfLogoContentChecker(WfSharedContentChecker):
    """Logo content checker"""

    def inner_check(self, request):
        output = super(WfLogoContentChecker, self).inner_check(request)
        translate = request.localizer.translate
        if not (self.context.image and self.context.image.get_size()):
            output.append(translate(MISSING_VALUE).format(field=translate(IWfLogo['image'].title),
                                                          message=translate(_("no image defined"))))
        if not self.context.url:
            output.append(translate(MISSING_VALUE).format(field=translate(IWfLogo['url'].title),
                                                          message=translate(_("no URL defined"))))
        return output


@adapter_config(context=(IWfLogo, IPyAMSUserLayer), provides=ICanonicalURL)
class WfLogoCanonicalUrlAdapter(ContextRequestAdapter):
    """Logo canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        if self.context.url:
            return self.context.url
        elif self.context.reference:
            target = self.context.target
            if target is not None:
                return canonical_url(target, self.request, view_name, query)
        return None


@adapter_config(context=(IWfLogo, IPyAMSUserLayer), provides=IRelativeURL)
class WfLogoRelativeUrlAdapter(ContextRequestAdapter):
    """Logo relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        if self.context.url:
            return self.context.url
        elif self.context.reference:
            target = self.context.target
            if target is not None:
                return relative_url(target, self.request, display_context, view_name, query)
        return None
