#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.imagemap.interfaces import IImageMapParagraph, IMAGEMAP_PARAGRAPH_TYPE
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm, IWidgetsSuffixViewletsManager
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import Viewlet, viewlet_config

from pyams_content import _


@viewlet_config(name='add-imagemap-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=600)
class ImagemapParagraphAddMenu(BaseParagraphAddMenu):
    """Image map paragraph add menu"""

    label = _("Image map...")
    label_css_class = 'fa fa-fw fa-location-arrow'
    url = 'add-imagemap-paragraph.html'
    paragraph_type = IMAGEMAP_PARAGRAPH_TYPE


@pagelet_config(name='add-imagemap-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-imagemap-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class ImagemapParagraphAddForm(BaseParagraphAddForm):
    """Image map paragraph add form"""

    legend = _("Add new image map")

    content_interface = IImageMapParagraph


@pagelet_config(name='properties.html', context=IImageMapParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IImageMapParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class ImagemapParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Image map paragraph edit form"""

    prefix = 'imagemap_properties.'

    @property
    def title(self):
        content = get_parent(self.context, IWfSharedContent)
        return II18n(content).query_attribute('title', request=self.request)

    legend = _("Edit paragraph properties")

    content_interface = IImageMapParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'reference' in changes.get(IImageMapParagraph, ()):
            output.setdefault('events', []).append(
                get_json_form_refresh_event(self.context, self.request, ImagemapParagraphInnerEditForm))
        return output


@adapter_config(context=(IImageMapParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IImageMapParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class ImagemapParagraphInnerEditForm(ImagemapParagraphPropertiesEditForm):
    """Image map properties inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()


@viewlet_config(name='imagemap-preview', context=IImageMapParagraph, layer=IPyAMSLayer,
                view=ImagemapParagraphInnerEditForm, manager=IWidgetsSuffixViewletsManager, weight=1)
@template_config(template='templates/imagemap-preview.pt', layer=IPyAMSLayer)
class ImagemapParagraphPreviewWidgetsSuffix(Viewlet):
    """Image map paragraph preview widgets suffix"""
