#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from persistent.mapping import PersistentMapping
from pyramid.threadlocal import get_current_registry
from z3c.form.interfaces import NOT_CHANGED
from zope.container.contained import Contained
from zope.interface import implementer, provider
from zope.lifecycleevent import ObjectModifiedEvent
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_content import _
from pyams_content.component.association.interfaces import IAssociationContainer
from pyams_content.component.extfile.interfaces import IExtFileContainerTarget
from pyams_content.component.links.interfaces import ILinkContainerTarget
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review.interfaces import IReviewTarget
from pyams_content.shared.common import SharedContent, WfSharedContent, WfSharedContentChecker, register_content_type
from pyams_content.shared.common.interfaces import IWfSharedContentFactory
from pyams_content.shared.imagemap.interfaces import IImageMap, IImageMapArea, IMAGEMAP_CONTENT_NAME, \
    IMAGEMAP_CONTENT_TYPE, IWfImageMap, IWfImageMapFactory
from pyams_i18n.interfaces import II18n, II18nManager
from pyams_i18n.property import I18nFileProperty
from pyams_utils.adapter import ContextAdapter, adapter_config


@implementer(IImageMapArea)
class ImageMapArea(Persistent, Contained):
    """Image map area class"""

    title = FieldProperty(IImageMapArea['title'])
    link = FieldProperty(IImageMapArea['link'])
    area = FieldProperty(IImageMapArea['area'])

    @property
    def svg_points(self):
        points = self.area.split(',')
        x = points[::2]
        y = points[1::2]
        return ' '.join(','.join(xy) for xy in zip(x, y))


@implementer(IWfImageMap, IExtFileContainerTarget, ILinkContainerTarget,
             IPreviewTarget, IReviewTarget)
class WfImageMap(WfSharedContent):
    """Base image map"""

    content_type = IMAGEMAP_CONTENT_TYPE
    content_name = IMAGEMAP_CONTENT_NAME

    handle_content_url = False
    handle_header = False
    handle_description = False

    _image = I18nFileProperty(IWfImageMap['image'])
    areas = FieldProperty(IWfImageMap['areas'])

    _index = 0

    def __init__(self):
        self.areas = PersistentMapping()

    @property
    def image(self):
        return self._image

    @image.setter
    def image(self, value):
        if value:
            for image in value.values():
                if image and (image is not NOT_CHANGED):
                    self.areas = PersistentMapping()
                    break
        self._image = value
        
    @image.deleter
    def image(self):
        del self._image

    def add_area(self, area):
        self._index += 1
        key = str(self._index)
        self.areas[key] = area
        locate(area, self, '++areas++{0}'.format(key))
        get_current_registry().notify(ObjectModifiedEvent(self))

    def remove_area(self, key):
        if key in self.areas:
            del self.areas[key]
            get_current_registry().notify(ObjectModifiedEvent(self))

    def get_association(self, area):
        return IAssociationContainer(self).get(area.link)


register_content_type(WfImageMap, shared_content=False)


@provider(IWfImageMapFactory)
@implementer(IImageMap)
class ImageMap(SharedContent):
    """Workflow managed image map class"""


@adapter_config(context=IWfImageMapFactory, provides=IWfSharedContentFactory)
def image_map_content_factory(context):
    return WfImageMap


@adapter_config(name='areas', context=IWfImageMap, provides=ITraversable)
class ImapemapAreasNamespace(ContextAdapter):
    """++areas++ namespace traverser"""

    def traverse(self, name, furtherpath=None):
        return self.context.areas[name]


@adapter_config(name='areas', context=IWfImageMap, provides=ISublocations)
class ImagemapAreasSublocations(ContextAdapter):
    """Image map area sublocations"""

    def sublocations(self):
        return self.context.areas.values()


@adapter_config(name='properties', context=IWfImageMap, provides=IContentChecker)
class WfImageMapContentChecker(WfSharedContentChecker):
    """Image map content checker"""

    def inner_check(self, request):
        output = super(WfImageMapContentChecker, self).inner_check(request)
        translate = request.localizer.translate
        missing_value = translate(MISSING_VALUE)
        missing_lang_value = translate(MISSING_LANG_VALUE)
        langs = II18nManager(self.context).get_languages()
        i18n = II18n(self.context)
        for lang in langs:
            data = i18n.get_attribute('image', lang, request)
            if not data:
                if len(langs) == 1:
                    output.append(missing_value.format(field=translate(IWfImageMap['image'].title)))
                else:
                    output.append(missing_lang_value.format(field=translate(IWfImageMap['image'].title),
                                                            lang=lang))
        if not self.context.areas:
            output.append(translate(ERROR_VALUE).format(field=translate(IWfImageMap['areas'].title),
                                                        message=translate(_("no area defined"))))
        return output
