#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.events import subscriber
from pyramid.view import view_config
from z3c.form import field
from z3c.form.interfaces import DISPLAY_MODE, IDataExtractedEvent
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Invalid, implementer

from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.zmi import WfModifiedContentColumnMixin
from pyams_content.shared.form.field import FormField
from pyams_content.shared.form.interfaces import IFormField, IFormFieldContainer, \
    IFormFieldContainerTarget, IFormFieldFactory
from pyams_content.zmi import pyams_content
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IFormSecurityContext
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_visibility
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_skin.interfaces.container import ITableWithActions
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, I18nValueColumn, SorterColumn, TrashColumn, \
    VisibilitySwitcherColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import ContainerAdminView


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='form-fields.menu', context=IFormFieldContainerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=95)
class FormFieldsMenu(MenuItem):
    """Form fields menu"""

    label = _("Form fields...")
    icon_class = 'fa-th-list'
    url = '#form-fields.html'


#
# Form fields container view
#

@implementer(ITableWithActions)
class FormFieldsContainerTable(ProtectedFormObjectMixin, BaseTable):
    """Form fields table"""

    prefix = 'form_fields'

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(IFormFieldContainer(self.context), self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-form-fields-order.json'
        }
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-form-field-visibility.json'

    @reify
    def values(self):
        return list(super(FormFieldsContainerTable, self).values)

    def render(self):
        if not self.values:
            translate = self.request.localizer.translate
            return translate(_("No currently defined form field."))
        return super(FormFieldsContainerTable, self).render()


@adapter_config(name='sorter', context=(IFormFieldContainerTarget, IPyAMSLayer,
                                        FormFieldsContainerTable),
                provides=IColumn)
class FormFieldsContainerSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Form fields container sorter column"""


@adapter_config(name='show-hide', context=(IFormFieldContainerTarget, IPyAMSLayer,
                                           FormFieldsContainerTable),
                provides=IColumn)
class FormFieldsContainerShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Form fields container visibility switcher column"""


@view_config(name='switch-form-field-visibility.json', context=IFormFieldContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def switch_form_field_visibility(request):
    """Set form field visibility"""
    return switch_element_visibility(request, IFormFieldContainer)


@adapter_config(context=FormFieldsContainerShowHideColumn, provides=IFormSecurityContext)
def show_hide_column_security_context_factory(column):
    """Show/hide column security context factory"""
    return column.table.context


@adapter_config(name='name', context=(IFormFieldContainerTarget, IPyAMSLayer,
                                      FormFieldsContainerTable),
                provides=IColumn)
class FormFieldsContainerNameColumn(I18nColumn, WfModifiedContentColumnMixin, GetAttrColumn):
    """Form fields container name column"""

    _header = _("Name")

    attrName = 'name'
    weight = 50


@adapter_config(name='label', context=(IFormFieldContainerTarget, IPyAMSLayer,
                                       FormFieldsContainerTable),
                provides=IColumn)
class FormFieldsContainerLabelColumn(I18nColumn, WfModifiedContentColumnMixin, I18nValueColumn):
    """Form fields container label column"""

    _header = _("Label")

    attrName = 'label'
    weight = 55


@adapter_config(name='type', context=(IFormFieldContainerTarget, IPyAMSLayer,
                                      FormFieldsContainerTable),
                provides=IColumn)
class FormFieldsContainerTypeColumn(I18nColumn, WfModifiedContentColumnMixin, GetAttrColumn):
    """Form fields container label column"""

    _header = _("Field type")

    weight = 60

    def getValue(self, obj):
        adapter = self.request.registry.queryUtility(IFormFieldFactory, name=obj.field_type)
        if adapter is not None:
            label = adapter.label
        else:
            label = _("-- unknown field type --")
        return self.request.localizer.translate(label)


@adapter_config(name='trash', context=(IFormFieldContainerTarget, IPyAMSLayer,
                                       FormFieldsContainerTable),
                provides=IColumn)
class FormFieldsContainerTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Form fields container trash column"""


@adapter_config(context=(IFormFieldContainerTarget, IPyAMSLayer, FormFieldsContainerTable),
                provides=IValues)
class FormFieldsContainerValues(ContextRequestViewAdapter):
    """Form fields container values"""

    @property
    def values(self):
        return IFormFieldContainer(self.context).values()


@pagelet_config(name='form-fields.html', context=IFormFieldContainerTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class FormFieldsContainerView(ContainerAdminView):
    """Form fields container view"""

    title = _("Form fields list")
    table_class = FormFieldsContainerTable


@view_config(name='set-form-fields-order.json', context=IFormFieldContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def set_form_fields_order(request):
    """Update form fields order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


#
# Form field views
#

@viewlet_config(name='add-form-field.action', context=IFormFieldContainerTarget, layer=IAdminLayer,
                view=FormFieldsContainerView, manager=IWidgetTitleViewletManager,
                permission=MANAGE_CONTENT_PERMISSION, weight=1)
class FormFieldAddAction(ProtectedFormObjectMixin, ToolbarAction):
    """Form field add action"""

    label = _("Add form field")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-form-field.html'
    modal_target = True


@pagelet_config(name='add-form-field.html', context=IFormFieldContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-form-field.json', context=IFormFieldContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class FormFieldAddForm(AdminDialogAddForm):
    """Form field add form"""

    legend = _("Add form field")
    icon_css_class = 'fa fa-fw fa-th-list'

    fields = field.Fields(IFormField).omit('__parent__', '__name__', 'visible')
    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return FormField()

    def add(self, object):
        IFormFieldContainer(self.context)[object.name] = object

    def nextURL(self):
        return '#form-fields.html'


@subscriber(IDataExtractedEvent, form_selector=FormFieldAddForm)
def handle_new_form_field_data_extraction(event):
    """Handle new form field form data extraction"""
    container = IFormFieldContainer(event.form.context)
    name = translate_string(event.data.get('name'), spaces='_', keep_chars='_')
    if name in container:
        event.form.widgets.errors += (Invalid(_("Specified name is already used!")),)
    event.data['name'] = name


@pagelet_config(name='properties.html', context=IFormField, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IFormField, layer=IPyAMSLayer)
class FormFieldPropertiesEditForm(AdminDialogEditForm):
    """Form field properties edit form"""

    prefix = 'field_properties.'

    @property
    def title(self):
        content = get_parent(self.context, IWfSharedContent)
        return II18n(content).query_attribute('title', request=self.request)

    legend = _("Edit form field properties")
    icon_class = 'fa fa-fw fa-th-list'

    fields = field.Fields(IFormField).omit('__parent__', '__name__', 'visible')
    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(FormFieldPropertiesEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if changes:
            container = get_parent(self.context, IFormFieldContainerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(container, self.request,
                                                 FormFieldsContainerTable, self.context))
        return output
