#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form.interfaces import IAddForm
from zope.annotation.interfaces import IAttributeAnnotatable
from zope.container.constraints import containers, contains
from zope.container.interfaces import IContained, IContainer
from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice, Int, Password, TextLine

from pyams_content.component.paragraph import IBaseParagraph
from pyams_content.shared.common.interfaces import ISharedContent, ISharedToolPortalContext, \
    IWfSharedContentPortalContext
from pyams_i18n.schema import I18nHTMLField, I18nTextField, I18nTextLineField
from pyams_utils.schema import MailAddressField, TextLineListField


__docformat__ = 'restructuredtext'

from pyams_content import _


FORM_CONTENT_TYPE = 'form'
FORM_CONTENT_NAME = _('Form')

FORM_FIELD_CONTAINER_KEY = 'pyams_content.shared.form_fields'


class IFormsManager(ISharedToolPortalContext):
    """Forms manager interface"""

    use_captcha = Bool(title=_("Use captcha?"),
                       description=_("Set default captcha settings"),
                       required=True,
                       default=False)

    default_captcha_client_key = TextLine(title=_("Default captcha site key"),
                                          description=_("This key is included into HTML code and "
                                                        "submitted with form data"),
                                          required=False)

    default_captcha_server_key = TextLine(title=_("Default captcha secret key"),
                                          description=_("This key is used to communicate with "
                                                        "Google's reCaptcha services"),
                                          required=False)

    def get_captcha_settings(self):
        """Get default captcha settings"""

    use_proxy = Bool(title=_("Use proxy server?"),
                     description=_("If a proxy server is required to access recaptcha services, "
                                   "please set them here"),
                     required=True,
                     default=False)

    proxy_proto = Choice(title=_("Protocol"),
                         description=_("If your server is behind a proxy, please set it's "
                                       "protocol here; HTTPS support is required for reCaptcha"),
                         required=False,
                         values=('http', 'https'),
                         default='http')

    proxy_host = TextLine(title=_("Host name"),
                          description=_("If your server is behind a proxy, please set it's "
                                        "address here; captcha verification requires HTTPS "
                                        "support..."),
                          required=False)

    proxy_port = Int(title=_("Port number"),
                     description=_("If your server is behind a proxy, plase set it's port "
                                   "number here"),
                     required=False,
                     default=8080)

    proxy_username = TextLine(title=_("Username"),
                              description=_("If your proxy server requires authentication, "
                                            "please set username here"),
                              required=False)

    proxy_password = Password(title=_("Password"),
                              description=_("If your proxy server requires authentication, "
                                            "please set password here"),
                              required=False)

    proxy_only_from = TextLine(title=_("Use proxy only from"),
                               description=_("If proxy usage is restricted to several domains "
                                             "names, you can set them here, separated by comas"),
                               required=False)

    def get_proxy_url(self, request):
        """Get proxy server URL"""


class IFormsManagerFactory(Interface):
    """Forms manager factory interface"""


class IFormField(IContained, IAttributeAnnotatable):
    """Form field interface"""

    containers('.IFormFieldContainer')

    name = TextLine(title=_("Field name"),
                    description=_("Field internal name; must be unique for a given form"),
                    required=True)

    field_type = Choice(title=_("Field type"),
                        description=_("Selected field type"),
                        vocabulary='PyAMS form field types',
                        required=True)

    label = I18nTextLineField(title=_("Label"),
                              description=_("User field label"),
                              required=True)

    description = I18nTextField(title=_("Description"),
                                description=_("Field description can be displayed as hint"),
                                required=False)

    placeholder = TextLine(title=_("Placeholder"),
                           description=_("Some field types like textline can display a "
                                         "placeholder"),
                           required=False)

    values = TextLineListField(title=_("Optional values"),
                               description=_("List of available values (for 'choice' and 'list' "
                                             "field types)"),
                               required=False)

    default = I18nTextLineField(title=_("Default value"),
                                description=_("Give default value if field type can use it"),
                                required=False)

    required = Bool(title=_("Required?"),
                    description=_("Select 'yes' to set field as mandatory"),
                    required=True,
                    default=False)

    visible = Bool(title=_("Visible?"),
                   description=_("Select 'no' to hide given field..."),
                   required=True,
                   default=True)


class IFormFieldFactory(Interface):
    """Form field factory interface"""

    label = Attribute("Factory label")
    weight = Attribute("Factory weight")

    def get_schema_field(self, field):
        """Get schema field matching given form field"""


class IFormFieldDataConverter(Interface):
    """Interface of a converter adapter which can be used to convert form data"""

    def convert(self, value):
        """Convert given input value"""


FORM_FIELDS_PARAGRAPH_TYPE = 'form-fields'
FORM_FIELDS_PARAGRAPH_NAME = _("Form fields")
FORM_FIELDS_PARAGRAPH_RENDERERS = 'PyAMS.paragraph.formfields.renderers'


class IFormFieldsParagraph(IBaseParagraph):
    """Form fields paragraph"""

    renderer = Choice(title=_("Form fields template"),
                      description=_("Presentation template used for this paragraph"),
                      vocabulary=FORM_FIELDS_PARAGRAPH_RENDERERS,
                      default='default')


class IFormFieldContainer(IContainer):
    """Form fields container interface"""

    contains(IFormField)

    def append(self, field):
        """Append given field to container"""

    def get_fields(self):
        """Get schema fields matching current fields"""

    def find_fields(self, factory):
        """Find fields matching given factory (defined by it's utility name)"""


class IFormFieldContainerInputForm(IAddForm):
    """Form field container input form marker interface"""


class IFormFieldContainerTarget(Interface):
    """Form fields container target marker interface"""


class IWfForm(IWfSharedContentPortalContext):
    """Form interface"""

    alt_title = I18nTextLineField(title=_("Alternate title"),
                                  description=_("If set, this title will be displayed in "
                                                "front-office instead of original title"),
                                  required=False)

    form_header = I18nTextField(title=_("Form header"),
                                description=_("This header is displayed just above form fields"),
                                required=False)

    user_title = I18nTextLineField(title=_("Form title"),
                                   description=_("If set, this title will be displayed above input "
                                                 "fields"),
                                   required=False)

    auth_only = Bool(title=_("Authenticated only?"),
                     description=_("If 'yes', only authenticated users will be able to see and "
                                   "submit form"),
                     required=True,
                     default=False)

    submit_label = I18nTextLineField(title=_("Submit button"),
                                     description=_("Label of form submit button"),
                                     required=True)

    submit_message = I18nHTMLField(title=_("Submit message"),
                                   description=_("This message will be displayed after form "
                                                 "submission"),
                                   required=True)

    handler = Choice(title=_("Form handler"),
                     description=_("Select how form data is transmitted"),
                     vocabulary='PyAMS form handlers')

    override_captcha = Bool(title=_("Override captcha settings?"),
                            description=_("If 'yes', you can define custom captcha keys here"),
                            required=False,
                            default=True)

    client_captcha_key = TextLine(title=_("Site key"),
                                  description=_("This key is included into HTML code and submitted "
                                                "with form data"),
                                  required=False)

    server_captcha_key = TextLine(title=_("Secret key"),
                                  description=_("This key is used to communicate with Google's "
                                                "reCaptcha services"),
                                  required=False)

    def get_captcha_settings(self):
        """Get form captcha settings"""

    rgpd_consent = Bool(title=_("Required RGPD consent?"),
                        description=_("If 'yes', an RGPD compliance warning will be displayed "
                                      "above form's submit button; form can't be submitted as long "
                                      "as the associated checkbox will not be checked explicitly "
                                      "by the user"),
                        required=True,
                        default=True)

    rgpd_warning = I18nTextField(title=_("RGPD consent text"),
                                 description=_("User consent must be explicit, and user must be "
                                               "warned about usage which will be made of submitted "
                                               "data; text samples are given below"),
                                 required=False)

    rgpd_user_rights = I18nHTMLField(title=_("RGPD user rights"),
                                     description=_("The internet user must be able to easily "
                                                   "revoke his consent later on, so it is "
                                                   "important to inform him how to proceed; below "
                                                   "are examples of possible formulations"),
                                     required=False)

    def query_handler(self, handler=None):
        """Get form handler utility"""


class IWfFormFactory(Interface):
    """Form factory interface"""


class IForm(ISharedContent):
    """Workflow managed form interface"""


#
# Form handler
#

class IFormHandler(Interface):
    """Form handler interface"""

    label = Attribute("Handler label")
    target_interface = Attribute("Handler target marker interface")
    handler_info = Attribute("Handler info interface")

    def handle(self, form, data, user_data):
        """Handle entered data

        :param form: input form
        :param data: raw form data
        :param user_data: user friendly form input data
        """


class IFormHandlerInfo(Interface):
    """Base handler info interface"""


class IMailtoHandlerInfo(IFormHandlerInfo):
    """Mailto form handler info interface"""

    service_name = I18nTextLineField(title=_("Service name"),
                                     description=_("This name will be used to prefix all email "
                                                   "messages sent by this contact form"),
                                     required=False)

    source_address = MailAddressField(title=_("Source address"),
                                      description=_("Mail address from which form data is sent"),
                                      required=False)

    source_name = TextLine(title=_("Source name"),
                           description=_("Name of mail data sender"),
                           required=False)

    target_address = MailAddressField(title=_("Recipient address"),
                                      description=_("Mail address to which form data is sent"),
                                      required=False)

    target_name = TextLine(title=_("Recipient name"),
                           description=_("Name of data recipient"),
                           required=False)

    notification_message = I18nHTMLField(title=_("Notification message"),
                                         description=_("This message will be sent to target "
                                                       "address defined just before; you can "
                                                       "include form fields values by including "
                                                       "their name, enclosed in {brackets}; if "
                                                       "you don't provide any template, message "
                                                       "will be filled only with the raw data "
                                                       "entered into this form"),
                                         required=False)

    confirm_message = I18nHTMLField(title=_("Confirmation message"),
                                    description=_("If an email address is provided in the form, "
                                                  "this message will be returned to the contact; "
                                                  "you can include all form fields values by "
                                                  "including their name, enclosed in {brackets}"),
                                    required=False)


class IMailtoHandlerTarget(IAttributeAnnotatable):
    """Mailto handler target marker interface"""
