#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
import datetime
from collections import OrderedDict

from persistent import Persistent
from pyramid_mailer.interfaces import IMailer
from zope.componentvocabulary.vocabulary import UtilityTerm, UtilityVocabulary
from zope.schema.fieldproperty import FieldProperty

from pyams_content.shared.form import IFormFieldContainer
from pyams_content.shared.form.interfaces import IFormHandler, IMailtoHandlerInfo, \
    IMailtoHandlerTarget
from pyams_i18n.interfaces import II18n
from pyams_mail.message import HTMLMessage
from pyams_security.interfaces import ISecurityManager
from pyams_security.interfaces.notification import INotificationSettings
from pyams_utils.adapter import adapter_config, get_adapter_weight, get_annotation_adapter
from pyams_utils.dict import escape_dict
from pyams_utils.factory import factory_config
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility, query_utility, utility_config
from pyams_utils.request import check_request, query_request
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'

from pyams_content import _


@vocabulary_config(name='PyAMS form handlers')
class FormHandlersVocabulary(UtilityVocabulary):
    """Form handlers vocabulary"""

    interface = IFormHandler

    def __init__(self, context, **kw):
        request = check_request()
        registry = request.registry
        translate = request.localizer.translate
        utils = [(None, translate(_("No selected handler...")))] + \
                [(name, translate(util.label))
                 for (name, util) in sorted(registry.getUtilitiesFor(self.interface),
                                            key=get_adapter_weight)]
        self._terms = OrderedDict((title, UtilityTerm(name, title)) for name, title in utils)

    def __iter__(self):
        return iter(self._terms.values())


#
# Mailto form handler
#

MAILTO_HANDLER_ANNOTATIONS_KEY = 'pyams_content.form.handler.mailto'


@factory_config(IMailtoHandlerInfo)
class MailtoFormHandlerInfo(Persistent):
    """Mailto form handler persistent info"""

    service_name = FieldProperty(IMailtoHandlerInfo['service_name'])
    source_address = FieldProperty(IMailtoHandlerInfo['source_address'])
    source_name = FieldProperty(IMailtoHandlerInfo['source_name'])
    target_address = FieldProperty(IMailtoHandlerInfo['target_address'])
    target_name = FieldProperty(IMailtoHandlerInfo['target_name'])
    notification_message = FieldProperty(IMailtoHandlerInfo['notification_message'])
    confirm_message = FieldProperty(IMailtoHandlerInfo['confirm_message'])


@adapter_config(context=IMailtoHandlerTarget, provides=IMailtoHandlerInfo)
def mailto_form_handler_factory(context):
    """Mailto form handler factory"""
    return get_annotation_adapter(context, MAILTO_HANDLER_ANNOTATIONS_KEY, IMailtoHandlerInfo)


@utility_config(name='mailto', provides=IFormHandler)
class MailtoFormHandler(object):
    """Mailto form handler"""

    label = _("Direct mailto form handler")
    weight = 10

    target_interface = IMailtoHandlerTarget
    handler_info = IMailtoHandlerInfo

    @staticmethod
    def get_oid():
        now = datetime.datetime.utcnow()
        return now.strftime('%y%m%d-%H%M%S')

    @staticmethod
    def build_message(template, data):
        data = escape_dict(data)
        if template:
            return template.format(**data)
        return '<br />'.join(('{}: {}'.format(k, v or '--') for k, v in data.items()))

    def handle(self, form, data, user_data):
        sm = get_utility(ISecurityManager)
        settings = INotificationSettings(sm)
        if not settings.enable_notifications:
            return
        mailer = query_utility(IMailer, name=settings.mailer)
        request = query_request()
        handler_info = self.handler_info(form)
        i18n_handler = II18n(handler_info)
        if handler_info.target_address:
            user_data['_reference'] = self.get_oid()
            body = self.build_message(
                i18n_handler.query_attribute('notification_message', request=request),
                user_data)
            target = '{} <{}>'.format(handler_info.target_name, handler_info.target_address) \
                if handler_info.target_name else handler_info.target_address
            message = HTMLMessage(
                subject='[{}] {}'.format(settings.subject_prefix,
                                         i18n_handler.query_attribute('service_name',
                                                                      request=request)),
                fromaddr='{} <{}>'.format(handler_info.source_name,
                                          handler_info.source_address),
                toaddr=target,
                html=body)
            mailer.send(message)
        fields = IFormFieldContainer(form)
        has_email, email_field = boolean_iter(fields.find_fields('mail'))
        email_address = data.get(next(email_field).name)
        confirm_message = i18n_handler.query_attribute('confirm_message',
                                                       request=request)
        if not (has_email and email_address and confirm_message):
            return
        message = HTMLMessage(
            subject='[{}] {}'.format(settings.subject_prefix, i18n_handler.query_attribute(
                'service_name', request=request)),
            fromaddr='{} <{}>'.format(handler_info.source_name, handler_info.source_address),
            toaddr=email_address,
            html=confirm_message.format(**user_data))
        mailer.send(message)
