#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from datetime import datetime

from pyramid.events import subscriber
from z3c.form import button, field
from z3c.form.browser.radio import RadioWidget
from z3c.form.interfaces import IDataExtractedEvent
from z3c.form.widget import FieldWidget
from zope.interface import Interface, Invalid
from zope.lifecycleevent import ObjectModifiedEvent
from zope.schema import Bool

from pyams_content import _
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.interfaces import CREATE_VERSION_PERMISSION, MANAGE_CONTENT_PERMISSION, PUBLISH_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IBaseSharedTool, IContributorRestrictions, IManagerRestrictions, \
    ISharedContent, IWfSharedContent, WfSharedContentNotificationEvent
from pyams_content.workflow import DELETED, DRAFT
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces import IFormLayer
from pyams_form.interfaces.form import IFormSuffixViewletsManager, IInnerSubForm, IWidgetsPrefixViewletsManager
from pyams_form.schema import CloseButton
from pyams_form.widget import widgettemplate_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.interfaces import ISecurityManager
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.registry import get_utility
from pyams_utils.text import text_to_html
from pyams_utils.timezone import gmtime, tztime
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_workflow.interfaces import AmbiguousTransitionError, IWorkflow, IWorkflowCommentInfo, IWorkflowInfo, \
    IWorkflowPublicationInfo, IWorkflowRequestUrgencyInfo, IWorkflowState, IWorkflowStateLabel, IWorkflowTransitionInfo, \
    IWorkflowVersions, MANUAL, SYSTEM
from pyams_workflow.zmi.transition import WorkflowContentTransitionForm
from pyams_zmi.form import InnerAdminAddForm


#
# Generic transition info
#

@viewlet_config(name='wf-transition-info', context=IWfSharedContent, layer=IPyAMSLayer,
                view=WorkflowContentTransitionForm, manager=IFormSuffixViewletsManager, weight=10)
@template_config(template='templates/wf-transition-info.pt')
class WorkflowContentTransitionFormInfo(Viewlet):
    """Publication request form info message"""

    @property
    def previous_step(self):
        translate = self.request.localizer.translate
        workflow = IWorkflow(self.context)
        state = IWorkflowState(self.context)
        registry = self.request.registry
        adapter = registry.queryMultiAdapter((workflow, self.request), IWorkflowStateLabel, name=state.state)
        if adapter is None:
            adapter = registry.queryAdapter(workflow, IWorkflowStateLabel, name=state.state)
        if adapter is None:
            adapter = registry.queryAdapter(workflow, IWorkflowStateLabel)
        if adapter is not None:
            state_label = adapter.get_label(self.context, request=self.request)
        else:
            state_label = translate(_("{state} {date}")).format(
                state=translate(workflow.get_state_label(state.state)),
                date=format_datetime(state.state_date, request=self.request))
        security = get_utility(ISecurityManager)
        return translate(_("{state} | by {principal}")).format(
            state=state_label,
            principal=security.get_principal(state.state_principal).title)

    @property
    def previous_message(self):
        workflow = IWorkflow(self.context)
        state = IWorkflowState(self.context)
        position = 0
        history_item = None
        trigger = SYSTEM
        while trigger != MANUAL:
            position -= 1
            history_item = state.history[position]
            if history_item.transition_id:
                try:
                    trigger = workflow.get_transition_by_id(history_item.transition_id).trigger
                except KeyError:
                    continue
            else:
                break
        if history_item:
            return text_to_html((history_item.comment or '').strip())

    @property
    def next_step(self):
        transition = self.__parent__.transition
        return self.request.localizer.translate(transition.user_data.get('next_step')) \
            if 'next_step' in transition.user_data else None


#
# Request publication form
#

class IPublicationRequestButtons(Interface):
    """Shared content publication request buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Request publication"))


@pagelet_config(name='wf-propose.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-propose.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationRequestForm(WorkflowContentTransitionForm):
    """Shared content publication request form"""

    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    @property
    def fields(self):
        pub_fields = ('publication_effective_date', 'push_end_date', 'publication_expiration_date')
        state = IWorkflowState(self.context)
        if state.version_id > 1:
            pub_fields += ('displayed_publication_date',)
        return field.Fields(IWorkflowTransitionInfo) + \
               field.Fields(IWorkflowPublicationInfo).select(*pub_fields) + \
               field.Fields(IWorkflowRequestUrgencyInfo) + \
               field.Fields(IWorkflowCommentInfo)

    buttons = button.Buttons(IPublicationRequestButtons)

    def updateWidgets(self, prefix=None):
        super(PublicationRequestForm, self).updateWidgets(prefix)
        pub_info = IWorkflowPublicationInfo(self.context)
        if 'publication_effective_date' in self.widgets:
            widget = self.widgets['publication_effective_date']
            widget.required = True
            widget.value = tztime(datetime.utcnow()).strftime('%d/%m/%y %H:%M')
        if ('push_end_date' in self.widgets) and pub_info.push_end_date:
            self.widgets['push_end_date'].value = tztime(pub_info.push_end_date).strftime('%d/%m/%y %H:%M')
        if ('publication_expiration_date' in self.widgets) and pub_info.publication_expiration_date:
            self.widgets['publication_expiration_date'].value = \
                tztime(pub_info.publication_expiration_date).strftime('%d/%m/%y %H:%M')
        if 'displayed_publication_date' in self.widgets:
            self.widgets['displayed_publication_date'].value = pub_info.displayed_publication_date

    def createAndAdd(self, data):
        data = data.get(self, data)
        pub_info = IWorkflowPublicationInfo(self.context)
        pub_info.publication_effective_date = data.get('publication_effective_date')
        pub_info.push_end_date = data.get('push_end_date')
        pub_info.publication_expiration_date = data.get('publication_expiration_date')
        if 'displayed_publication_date' in data:
            pub_info.displayed_publication_date = data.get('displayed_publication_date')
        return super(PublicationRequestForm, self).createAndAdd(data)


@subscriber(IDataExtractedEvent, form_selector=PublicationRequestForm)
def handle_publication_request_form_data_extraction(event):
    """Handle publication request form data extraction"""
    data = event.data
    if not data.get('publication_effective_date'):
        event.form.widgets.errors += (Invalid(_("Publication start date is required")),)


@viewlet_config(name='wf-propose-message', context=IWfSharedContent, layer=IPyAMSLayer, view=PublicationRequestForm,
                manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-propose-message.pt')
class PublicationRequestFormMessage(Viewlet):
    """Publication request form info message"""


@viewlet_config(name='wf-propose-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRequestForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationRequestFormWarning(Viewlet):
    """Publication request form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Cancel publication request form
#

class IPublicationRequestCancelButtons(Interface):
    """Shared content publication request cancel buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Cancel publication request"))


@pagelet_config(name='wf-cancel-propose.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-cancel-propose.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationRequestCancelForm(WorkflowContentTransitionForm):
    """Shared content publication request cancel form"""

    buttons = button.Buttons(IPublicationRequestCancelButtons)


@viewlet_config(name='wf-cancel-propose-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRequestCancelForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-cancel-propose-message.pt')
class PublicationRequestCancelFormMessage(Viewlet):
    """Publication request cancel form info message"""


@viewlet_config(name='wf-cancel-propose-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRequestCancelForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationRequestCancelFormWarning(Viewlet):
    """Publication request cancel form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Refuse publication form
#

class IPublicationRequestRefuseButtons(Interface):
    """Shared content publication request refuse buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Refuse publication request"))


@pagelet_config(name='wf-refuse.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=PUBLISH_CONTENT_PERMISSION)
@ajax_config(name='wf-refuse.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=PUBLISH_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationRequestRefuseForm(WorkflowContentTransitionForm):
    """Shared content publication request refuse form"""

    buttons = button.Buttons(IPublicationRequestRefuseButtons)

    def updateWidgets(self, prefix=None):
        super(PublicationRequestRefuseForm, self).updateWidgets(prefix)
        if 'comment' in self.widgets:
            self.widgets['comment'].required = True


@subscriber(IDataExtractedEvent, form_selector=PublicationRequestRefuseForm)
def handle_publication_request_refuse_form_data_extraction(event):
    """Handle publication request refuse form data extraction"""
    comment = (event.data.get('comment') or '').strip()
    if not comment:
        event.form.widgets.errors += (Invalid(_("A comment is required")),)


@viewlet_config(name='wf-refuse-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRequestRefuseForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-refuse-propose-message.pt')
class PublicationRequestRefuseFormMessage(Viewlet):
    """Publication request refuse form info message"""


@viewlet_config(name='wf-refuse-operator-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRequestRefuseForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-operator-warning.pt')
class PublicationRequestRefuseFormWarning(Viewlet):
    """Publication request refuse form warning message"""

    def __new__(cls, context, request, view, manager):
        state = IWorkflowState(context)
        if state.state_principal in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publish form
#

class IPublicationButtons(Interface):
    """Shared content publication buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Publish"))


@pagelet_config(name='wf-publish.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=PUBLISH_CONTENT_PERMISSION)
@ajax_config(name='wf-publish.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=PUBLISH_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationForm(WorkflowContentTransitionForm):
    """Shared content publication form"""

    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    @property
    def fields(self):
        pub_fields = ('publication_effective_date', 'push_end_date',
                      'publication_expiration_date')
        state = IWorkflowState(self.context)
        if state.version_id > 1:
            pub_fields += ('displayed_publication_date',)
        return field.Fields(IWorkflowTransitionInfo) + \
            field.Fields(IWorkflowPublicationInfo).select(*pub_fields) + \
            field.Fields(IWorkflowCommentInfo)

    buttons = button.Buttons(IPublicationButtons)

    def updateWidgets(self, prefix=None):
        super(PublicationForm, self).updateWidgets(prefix)
        pub_info = IWorkflowPublicationInfo(self.context)
        if 'publication_effective_date' in self.widgets:
            widget = self.widgets['publication_effective_date']
            widget.required = True
            now = gmtime(datetime.utcnow())
            if pub_info.publication_effective_date:
                widget.value = tztime(max(now, pub_info.publication_effective_date)).strftime('%d/%m/%y %H:%M')
            else:
                widget.value = tztime(now).strftime('%d/%m/%y %H:%M')
        if ('push_end_date' in self.widgets) and pub_info.push_end_date:
            self.widgets['push_end_date'].value = \
                tztime(pub_info.push_end_date).strftime('%d/%m/%y %H:%M')
        if ('publication_expiration_date' in self.widgets) and pub_info.publication_expiration_date:
            self.widgets['publication_expiration_date'].value = \
                tztime(pub_info.publication_expiration_date).strftime('%d/%m/%y %H:%M')
        if 'displayed_publication_date' in self.widgets:
            self.widgets['displayed_publication_date'].value = pub_info.displayed_publication_date

    def createAndAdd(self, data):
        data = data.get(self, data)
        context = self.context
        request = self.request
        pub_info = IWorkflowPublicationInfo(context)
        pub_info.publication_effective_date = data.get('publication_effective_date')
        pub_info.push_end_date = data.get('push_end_date')
        pub_info.publication_expiration_date = data.get('publication_expiration_date')
        if 'displayed_publication_date' in data:
            pub_info.displayed_publication_date = data.get('displayed_publication_date')
        now = gmtime(datetime.utcnow())
        if pub_info.publication_effective_date <= now:
            # immediate publication
            result = super(PublicationForm, self).createAndAdd(data)
            if IWorkflowState(context).state in IWorkflow(context).published_states:
                request.registry.notify(WfSharedContentNotificationEvent(context, request))
            return result
        else:
            # delayed publication: we schedule a publication task
            transition = self.transition.user_data.get('prepared_transition')
            if transition is None:
                raise AmbiguousTransitionError("This workflow doesn't support pre-publication!")
            info = IWorkflowInfo(context)
            info.fire_transition(transition.transition_id, comment=data.get('comment'))
            info.fire_automatic()
            IWorkflowState(context).state_urgency = data.get('urgent_request') or False
            request.registry.notify(ObjectModifiedEvent(context))
            return info


@subscriber(IDataExtractedEvent, form_selector=PublicationForm)
def handle_publication_form_data_extraction(event):
    """Handle publication form data extraction"""
    data = event.data
    if not data.get('publication_effective_date'):
        event.form.widgets.errors += (Invalid(_("Publication start date is required")),)


@viewlet_config(name='wf-publish-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-publish-message.pt')
class PublicationFormMessage(Viewlet):
    """Shared content publication form info message"""


@viewlet_config(name='wf-publish-operator-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-operator-warning.pt')
class PublicationFormWarning(Viewlet):
    """Shared content publication form warning message"""

    def __new__(cls, context, request, view, manager):
        state = IWorkflowState(context)
        if state.state_principal in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Pre-publication cancel form
#

class IPublicationCancelButtons(Interface):
    """Shared content publication cancel buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Cancel publication"))


@pagelet_config(name='wf-cancel-publish.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-cancel-publish.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationCancelForm(WorkflowContentTransitionForm):
    """Shared content publication cancel form"""

    fields = field.Fields(IWorkflowTransitionInfo) + \
             field.Fields(IWorkflowCommentInfo)
    buttons = button.Buttons(IPublicationCancelButtons)


@viewlet_config(name='wf-cancel-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationCancelForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-cancel-publish-message.pt')
class PublicationCancelFormMessage(Viewlet):
    """Publication cancel form info message"""


@viewlet_config(name='wf-cancel-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationCancelForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationCancelFormWarning(Viewlet):
    """Publication cancel form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publication retire request form
#

class IPublicationRetireRequestButtons(Interface):
    """Shared content publication retire request buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Request retire"))


@pagelet_config(name='wf-retiring.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-retiring.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationRetireRequestForm(WorkflowContentTransitionForm):
    """Shared content publication request refuse form"""

    fields = field.Fields(IWorkflowTransitionInfo) + \
             field.Fields(IWorkflowRequestUrgencyInfo) + \
             field.Fields(IWorkflowCommentInfo)
    buttons = button.Buttons(IPublicationRetireRequestButtons)

    def updateWidgets(self, prefix=None):
        super(PublicationRetireRequestForm, self).updateWidgets(prefix)
        if 'comment' in self.widgets:
            self.widgets['comment'].required = True


@subscriber(IDataExtractedEvent, form_selector=PublicationRetireRequestForm)
def handle_publication_retire_request_form_data_extraction(event):
    """Handle publication retire request form data extraction"""
    comment = (event.data.get('comment') or '').strip()
    if not comment:
        event.form.widgets.errors += (Invalid(_("A comment is required")),)


@viewlet_config(name='wf-retiring-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRetireRequestForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-retiring-message.pt')
class PublicationRetireRequestFormMessage(Viewlet):
    """Publication retire request form info message"""


@viewlet_config(name='wf-retiring-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRetireRequestForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationRetireRequestFormWarning(Viewlet):
    """Publication retire request form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publication retire cancel form
#

class IPublicationRetireCancelButtons(Interface):
    """Shared content publication retire request cancel buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Cancel retire request"))


@pagelet_config(name='wf-cancel-retiring.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-cancel-retiring.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationRetireCancelForm(WorkflowContentTransitionForm):
    """Shared content publication retire request cancel form"""

    buttons = button.Buttons(IPublicationRetireCancelButtons)


@viewlet_config(name='wf-cancel-retiring-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRetireCancelForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-cancel-retiring-message.pt')
class PublicationRetireCancelFormMessage(Viewlet):
    """Publication retire request form info message"""


@viewlet_config(name='wf-cancel-retiring-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRetireCancelForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationRetireCancelFormWarning(Viewlet):
    """Publication retire request cancel form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publication retire form
#

class IPublicationRetireButtons(Interface):
    """Shared content publication retire buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Retire"))


@pagelet_config(name='wf-retire.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=PUBLISH_CONTENT_PERMISSION)
@ajax_config(name='wf-retire.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=PUBLISH_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationRetireForm(WorkflowContentTransitionForm):
    """Shared content publication retire form"""

    buttons = button.Buttons(IPublicationRetireButtons)


@viewlet_config(name='wf-retire-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRetireForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-retire-message.pt')
class PublicationRetireFormMessage(Viewlet):
    """Publication retire form info message"""


@viewlet_config(name='wf-retire-operator-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationRetireForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-operator-warning.pt')
class PublicationRetireFormWarning(Viewlet):
    """Publication retire form warning message"""

    def __new__(cls, context, request, view, manager):
        state = IWorkflowState(context)
        if state.state_principal in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publication archive request form
#

class IPublicationArchiveRequestButtons(Interface):
    """Shared content publication archive request buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Request archive"))


@pagelet_config(name='wf-archiving.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-archiving.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationArchiveRequestForm(WorkflowContentTransitionForm):
    """Shared content publication request archive form"""

    fields = field.Fields(IWorkflowTransitionInfo) + \
             field.Fields(IWorkflowRequestUrgencyInfo) + \
             field.Fields(IWorkflowCommentInfo)
    buttons = button.Buttons(IPublicationArchiveRequestButtons)


@viewlet_config(name='wf-archiving-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationArchiveRequestForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-archiving-message.pt')
class PublicationArchiveRequestFormMessage(Viewlet):
    """Publication archive request form info message"""


@viewlet_config(name='wf-archiving-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationArchiveRequestForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationArchiveRequestFormWarning(Viewlet):
    """Publication archive request form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publication archive cancel form
#

class IPublicationArchiveCancelButtons(Interface):
    """Shared content publication archive request cancel buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Cancel archive request"))


@pagelet_config(name='wf-cancel-archiving.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-cancel-archiving.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationArchiveCancelForm(WorkflowContentTransitionForm):
    """Shared content publication archive request cancel form"""

    buttons = button.Buttons(IPublicationArchiveCancelButtons)


@viewlet_config(name='wf-cancel-archiving-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationArchiveCancelForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-cancel-archiving-message.pt')
class PublicationArchiveCancelFormMessage(Viewlet):
    """Publication archive cancel form info message"""


@viewlet_config(name='wf-cancel-archiving-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationArchiveCancelForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class PublicationArchiveCancelFormWarning(Viewlet):
    """Publication archive cancel form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Publication archive form
#

class IPublicationArchiveButtons(Interface):
    """Shared content publication archive buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Archive"))


@pagelet_config(name='wf-archive.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=PUBLISH_CONTENT_PERMISSION)
@ajax_config(name='wf-archive.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=PUBLISH_CONTENT_PERMISSION, base=AJAXAddForm)
class PublicationArchiveForm(WorkflowContentTransitionForm):
    """Shared content publication archive form"""

    buttons = button.Buttons(IPublicationArchiveButtons)


@viewlet_config(name='wf-archive-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationArchiveForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-archive-message.pt')
class PublicationArchiveFormMessage(Viewlet):
    """Publication archive form info message"""


@viewlet_config(name='wf-archive-operator-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=PublicationArchiveForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-operator-warning.pt')
class PublicationArchiveFormWarning(Viewlet):
    """Publication archive form warning message"""

    def __new__(cls, context, request, view, manager):
        state = IWorkflowState(context)
        if state.state_principal in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Clone form
#

class ISharedContentCloneButtons(Interface):
    """Shared content clone buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Create new version"))


@pagelet_config(name='wf-clone.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=CREATE_VERSION_PERMISSION)
@ajax_config(name='wf-clone.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=CREATE_VERSION_PERMISSION, base=AJAXAddForm)
class SharedContentCloneForm(WorkflowContentTransitionForm):
    """Shared content clone form"""

    buttons = button.Buttons(ISharedContentCloneButtons)

    def createAndAdd(self, data):
        data = data.get(self, data)
        info = IWorkflowInfo(self.context)
        return info.fire_transition_toward(DRAFT, comment=data.get('comment'))

    def get_ajax_output(self, changes):
        return {
            'status': 'redirect',
            'location': absolute_url(changes, self.request, 'admin#properties.html')
        }


@viewlet_config(name='wf-clone-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=SharedContentCloneForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-clone-message.pt')
class SharedContentCloneFormMessage(Viewlet):
    """Shared content clone form info message"""


@viewlet_config(name='wf-clone-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=SharedContentCloneForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class SharedContentCloneFormWarning(Viewlet):
    """Shared content clone form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Delete form
#

class ISharedContentDeleteButtons(Interface):
    """Shared content delete form buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    action = button.Button(name='action', title=_("Delete version"))


@pagelet_config(name='wf-delete.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='wf-delete.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AJAXAddForm)
class SharedContentDeleteForm(WorkflowContentTransitionForm):
    """Shared content delete form"""

    buttons = button.Buttons(ISharedContentDeleteButtons)

    __target = None

    @property
    def fields(self):
        fields = super(SharedContentDeleteForm, self).fields
        state = IWorkflowState(self.context)
        if state.version_id == 1:  # content deletion
            fields = fields.omit('comment')
        return fields

    @property
    def legend(self):
        state = IWorkflowState(self.context)
        if state.version_id == 1:  # content deletion
            return self.request.localizer.translate(_("Delete content"))
        else:
            return super(SharedContentDeleteForm, self).legend

    def updateActions(self):
        super(SharedContentDeleteForm, self).updateActions()
        if 'action' in self.actions:
            state = IWorkflowState(self.context)
            if state.version_id == 1:  # remove the first and only version => remove all
                self.actions['action'].title = _("Delete definitively")

    def createAndAdd(self, data):
        data = data.get(self, data)
        state = IWorkflowState(self.context)
        if state.version_id == 1:  # remove the first and only version => remove all
            content = get_parent(self.context, ISharedContent)
            self.__target = get_parent(content, IBaseSharedTool)
            del content.__parent__[content.__name__]
        else:
            versions = IWorkflowVersions(self.context)
            versions.remove_version(state.version_id, state=DELETED, comment=data.get('comment'))
            self.__target = versions.get_last_versions(count=1)[0]

    def get_ajax_output(self, changes):
        return {
            'status': 'redirect',
            'location': absolute_url(self.__target, self.request, 'admin')
        }


@viewlet_config(name='wf-delete-message', context=IWfSharedContent, layer=IPyAMSLayer,
                view=SharedContentDeleteForm, manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-delete-message.pt')
class SharedContentDeleteFormMessage(Viewlet):
    """Shared content delete form info message"""

    @property
    def version_id(self):
        state = IWorkflowState(self.context)
        return state.version_id


@viewlet_config(name='wf-delete-owner-warning', context=IWfSharedContent, layer=IPyAMSLayer,
                view=SharedContentDeleteForm, manager=IWidgetsPrefixViewletsManager, weight=30)
@template_config(template='templates/wf-owner-warning.pt')
class SharedContentDeleteFormWarning(Viewlet):
    """Shared content delete form warning message"""

    def __new__(cls, context, request, view, manager):
        if request.principal.id in context.owner:
            return None
        return Viewlet.__new__(cls)


#
# Contributor checks before publication request
#

class IContributorChecks(Interface):
    """Contributor checks interface"""

    preview = Bool(title=_("Previewed content?"),
                   required=True,
                   default=False)

    check = Bool(title=_("Verified content?"),
                 required=True,
                 default=False)


@widgettemplate_config(mode='input', template='templates/preview-input.pt', layer=IFormLayer)
class PreviewWidget(RadioWidget):
    """Preview check widget"""


def PreviewFieldWidget(field, request):
    return FieldWidget(field, PreviewWidget(request))


@widgettemplate_config(mode='input', template='templates/check-input.pt', layer=IFormLayer)
class CheckWidget(RadioWidget):
    """Control check widget"""


def CheckFieldWidget(field, request):
    return FieldWidget(field, CheckWidget(request))


@adapter_config(name='contributor-checks',
                context=(IWfSharedContent, IPyAMSLayer, PublicationRequestForm),
                provides=IInnerSubForm)
@adapter_config(name='contributor-checks',
                context=(IWfSharedContent, IPyAMSLayer, PublicationForm),
                provides=IInnerSubForm)
class ContributorChecksForm(InnerAdminAddForm):
    """Contributor checks form"""

    def __new__(cls, context, request, form):
        principal_id = request.principal.id
        restrictions = IManagerRestrictions(context, None)
        if restrictions is not None:
            manager_restrictions = restrictions.get_restrictions(principal_id)
            if (manager_restrictions is not None) and \
               (manager_restrictions.check_access(context, request=request)) and \
               (not manager_restrictions.publication_checks):
                return None
        if isinstance(form, PublicationRequestForm):
            restrictions = IContributorRestrictions(context, None)
            if restrictions is not None:
                contributor_restrictions = restrictions.get_restrictions(principal_id)
                if (contributor_restrictions is not None) and \
                   (not contributor_restrictions.publication_checks):
                    return None
        return InnerAdminAddForm.__new__(cls)

    legend = _("Prior checks")
    legend_class = 'inner bold text-danger'

    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    @property
    def fields(self):
        fields = field.Fields(IContributorChecks)
        if not IPreviewTarget.providedBy(self.context):
            fields = fields.omit('preview')
        else:
            fields['preview'].widgetFactory = PreviewFieldWidget
        fields['check'].widgetFactory = CheckFieldWidget
        return fields


@subscriber(IDataExtractedEvent, form_selector=ContributorChecksForm)
def handle_contributor_checks_data_extraction(event):
    """Handle extraction of contributor checks"""
    data = event.data
    can_preview = IPreviewTarget.providedBy(event.form.context)
    if can_preview and (not (data.get('preview') and data.get('check'))):
        event.form.widgets.errors += (Invalid(_("You must confirm that you previewed and checked this content before "
                                                "requesting publication!!")),)
    elif (not can_preview) and not data.get('check'):
        event.form.widgets.errors += (Invalid(_("You must confirm that you checked this content before "
                                                "requesting publication!!")),)
