#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq, NotAny
from z3c.form import button, field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import Interface
from zope.intid import IIntIds
from zope.lifecycleevent import ObjectModifiedEvent
from zope.schema import Choice, Bool

from pyams_catalog.query import CatalogResultSet
from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.shared.common import CONTENT_TYPES
from pyams_content.shared.common.interfaces import SHARED_TOOL_WORKFLOW_STATES_VOCABULARY
from pyams_content.shared.common.interfaces.types import DATA_TYPES_VOCABULARY, ITypedSharedTool
from pyams_content.shared.common.zmi.types import TypedSharedToolTypesView
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.schema import CloseButton
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.schema import Principal
from pyams_skin.interfaces.viewlet import IContextActions
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.registry import get_utility
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow
from pyams_zmi.form import AdminDialogAddForm

from pyams_content import _


#
# Typed shared content batch update form
#

@viewlet_config(name='update-datatype.html', context=ITypedSharedTool, layer=IPyAMSLayer,
                view=TypedSharedToolTypesView, manager=IContextActions, permission=MANAGE_TOOL_PERMISSION, weight=10)
class TypedSharedToolDatatypeUpdateMenu(ToolbarMenuItem):
    """Typed shared tool contents datatype update menu"""

    label = _("Update contents data type...")
    label_css_class = 'fa fa-fw fa-folder-o'

    url = 'update-datatype.html'
    modal_target = True


class ITypedSharedToolDatatypeUpdateInfo(Interface):
    """Typed shared tool contents datatype update form interface"""

    owner = Principal(title=_("Owner"),
                      required=False)

    source_datatype = Choice(title=_("Source data type"),
                             description=_("Current source data type to be updated"),
                             vocabulary=DATA_TYPES_VOCABULARY,
                             required=False)

    source_status = Choice(title=_("Source status"),
                           description=_("Current status to be updated"),
                           vocabulary=SHARED_TOOL_WORKFLOW_STATES_VOCABULARY,
                           required=False)

    ignore_archives = Bool(title=_("Ignore archives?"),
                           description=_("If 'yes', archives will not be updated"),
                           required=False,
                           default=True)

    target_datatype = Choice(title=_("Target data type"),
                             description=_("Target data type to which contents may be updated"),
                             vocabulary=DATA_TYPES_VOCABULARY,
                             required=False)


class ITypedSharedToolDatatypeUpdateButtons(Interface):
    """Typed shared tool contents datatype update form buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    update = button.Button(name='update', title=_("Update contents data type"))


@pagelet_config(name='update-datatype.html', context=ITypedSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='update-datatype.json', context=ITypedSharedTool, layer=IPyAMSLayer, base=AJAXAddForm)
class TypedSharedToolDatatypeUpdateForm(AdminDialogAddForm):
    """Typed shared tool content datatype update form"""

    legend = _("Contents data type update")
    icon_css_class = 'fa fa-fw fa-folder-o'

    @property
    def fields(self):
        fields = field.Fields(ITypedSharedToolDatatypeUpdateInfo)
        fields['ignore_archives'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    buttons = button.Buttons(ITypedSharedToolDatatypeUpdateButtons)

    edit_permission = MANAGE_TOOL_PERMISSION

    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    def updateWidgets(self, prefix=None):
        super(TypedSharedToolDatatypeUpdateForm, self).updateWidgets(prefix)
        if 'source_datatype' in self.widgets:
            self.widgets['source_datatype'].prompt = True
            self.widgets['source_datatype'].promptMessage = _("(select all datatypes)")
        if 'source_status' in self.widgets:
            self.widgets['source_status'].prompt = True
            self.widgets['source_status'].promptMessage = _("(select all status)")
        if 'target_datatype' in self.widgets:
            self.widgets['target_datatype'].prompt = True
            self.widgets['target_datatype'].promptMessage = _("Empty datatype")

    def updateActions(self):
        super(TypedSharedToolDatatypeUpdateForm, self).updateActions()
        if 'update' in self.actions:
            self.actions['update'].addClass('btn-primary')

    def createAndAdd(self, data):
        data = data.get(self, data)
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = Eq(catalog['parents'], intids.register(self.context)) & \
                 Any(catalog['content_type'], CONTENT_TYPES.keys())
        owner = data.get('owner')
        if owner:
            params &= Eq(catalog['role:owner'], owner)
        datatype = data.get('source_datatype')
        if datatype:
            params &= Eq(catalog['data_type'], datatype)
        status = data.get('source_status')
        if status:
            params &= Eq(catalog['workflow_state'], status)
        ignore_archives = data.get('ignore_archives', False)
        if ignore_archives:
            workflow = IWorkflow(self.context)
            params &= NotAny(catalog['workflow_state'], workflow.archived_states)
        notify = self.request.registry.notify
        datatype = data.get('target_datatype')
        updates = 0
        for content in CatalogResultSet(CatalogQuery(catalog).query(params)):
            if datatype != content.data_type:
                content.data_type = datatype
                notify(ObjectModifiedEvent(content))
                updates += 1
        return updates

    def get_ajax_output(self, changes):
        translate = self.request.localizer.translate
        if changes:
            return {
                'status': 'success',
                'message': translate(_("Update successful. "
                                       "{count} contents have been updated")).format(count=changes)
            }
        else:
            return {
                'status': 'info',
                'message': translate(_("Update finished. No content has been updated..."))
            }
