#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

import json

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.table.interfaces import IColumn, IValues

from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.shared.common.interfaces.types import ITypedDataManager, ITypedSharedTool
from pyams_content.zmi import pyams_content
from pyams_form.security import ProtectedFormObjectMixin
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import delete_container_element
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import ActionColumn, BaseTable, NameColumn, SorterColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuDivider, MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import ContainerAdminView

from pyams_content import _


@viewlet_config(name='data-types.divider', context=ITypedSharedTool, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=599)
class TypedSharedToolTyesMenuDivider(MenuDivider):
    """Typed shared tool types menu divider"""


@viewlet_config(name='data-types.menu', context=ITypedSharedTool, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=600)
class TypedSharedToolTypesMenu(MenuItem):
    """Typed shared tool types menu"""

    label = _("Data types")
    icon_class = 'fa-folder-o'
    url = '#data-types.html'


#
# Typed shared data types manager target views
#

class TypedSharedToolTypesTable(ProtectedFormObjectMixin, BaseTable):
    """Typed shared tool types table"""

    prefix = 'types'

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(ITypedDataManager(self.context), self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-types-order.json'
        }
        return attributes

    @reify
    def values(self):
        return list(ITypedDataManager(self.context).values())

    def render(self):
        if not self.values:
            translate = self.request.localizer.translate
            return translate(_("No currently defined data type."))
        return super().render()


@adapter_config(name='sorter', context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Typed shared tool types sorter column"""


@view_config(name='set-types-order.json', context=ITypedDataManager, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def set_data_types_order(request):
    """Update data types order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='name', context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesNameColumn(NameColumn):
    """Typed shared tool types name column"""

    _header = _("Data type label")

    def renderCell(self, item):
        return '<span data-ams-stop-propagation="true" ' \
               '      data-ams-click-handler="MyAMS.skin.switchCellContent" ' \
               '      data-ams-switch-handler="get-subtypes-table.json" ' \
               '      data-ams-switch-target=".subtypes">' \
               '    <span class="small hint" title="{hint}" data-ams-hint-gravity="e">' \
               '        <i class="fa fa-plus-square-o switch"></i>' \
               '    </span>' \
               '</span>&nbsp;&nbsp;&nbsp;<span class="title">{title}</span>' \
               '<div class="inner-table-form subtypes margin-x-10 margin-bottom-0 padding-left-5"></div>'.format(
            hint=self.request.localizer.translate(_("Click to see subtypes")),
            title=super(TypedSharedToolTypesNameColumn, self).renderCell(item))


@adapter_config(name='paragraphs', context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesParagraphsColumn(ActionColumn):
    """Typed shared tool types paragraphs column"""

    weight = 100

    icon_class = 'fa fa-fw fa-paragraph'
    icon_hint = _("Default paragraphs")

    url = 'paragraphs-dialog.html'
    modal_target = True

    permission = MANAGE_TOOL_PERMISSION


@adapter_config(name='associations',
                context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesAssociationsColumn(ActionColumn):
    """Typed shared tool types associations column"""

    weight = 110

    icon_class = 'fa fa-fw fa-link'
    icon_hint = _("Default associations")

    url = 'associations-dialog.html'
    modal_target = True

    permission = MANAGE_TOOL_PERMISSION


@adapter_config(name='tags',
                context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesTagsColumn(ActionColumn):
    """Typed shared tool types tags column"""

    weight = 115

    icon_class = 'fa fa-fw fa-tag'
    icon_hint = _("Default tags")

    url = 'tags-dialog.html'
    modal_target = True

    permission = MANAGE_TOOL_PERMISSION


@adapter_config(name='themes',
                context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesThemesColumn(ActionColumn):
    """Typed shared tool types themes column"""

    weight = 120

    icon_class = 'fa fa-fw fa-tags'
    icon_hint = _("Default themes")

    url = 'themes-dialog.html'
    modal_target = True

    permission = MANAGE_TOOL_PERMISSION


@adapter_config(name='trash', context=(ITypedSharedTool, IPyAMSLayer, TypedSharedToolTypesTable),
                provides=IColumn)
class TypedSharedToolTypesTrashColumn(TrashColumn):
    """Typed shared tool types trash column"""

    permission = MANAGE_TOOL_PERMISSION


@view_config(name='delete-element.json', context=ITypedDataManager, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def delete_data_type(request):
    """Data type delete view"""
    return delete_container_element(request, ignore_permission=True)


@pagelet_config(name='data-types.html', context=ITypedSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
class TypedSharedToolTypesView(ContainerAdminView):
    """Typed shared tool types view"""

    title = _("Content data types")
    table_class = TypedSharedToolTypesTable
