#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import field
from z3c.form.interfaces import DISPLAY_MODE, IDataConverter
from zope.interface import Interface

from pyams_content.shared.common.interfaces import IWfSharedContent, IWfSharedContentRoles
from pyams_form.interfaces.form import IInnerSubForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.utility import get_principal
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.interfaces.viewlet import IToolbarViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.timezone import tztime
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo, IWorkflowState, \
    IWorkflowStateHistoryItem, IWorkflowVersions
from pyams_zmi.form import AdminDialogDisplayForm, InnerAdminAddForm, InnerAdminDisplayForm
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='summary.action',
                context=IWfSharedContent, layer=IAdminLayer, view=Interface,
                manager=IToolbarViewletManager, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
class SummaryAction(ToolbarAction):
    """Summary action"""

    hint = _("Content information")

    group_css_class = 'btn-group margin-right-10'
    label_css_class = 'fa fa-fw fa-info txt-color-info'
    css_class = 'btn btn-xs btn-default btn-fa-rounded border-color-info hint'

    url = 'summary.html'
    modal_target = True


@pagelet_config(name='summary.html',
                context=IWfSharedContent, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedContentSummaryForm(AdminDialogDisplayForm):
    """Shared content summary form"""

    legend = _("Display content summary")
    css_class = 'ams-form form-horizontal form-tight'
    dialog_class = 'modal-large'

    fields = field.Fields(Interface)


@adapter_config(name='dublincore-summary',
                context=(IWfSharedContent, IPyAMSLayer, SharedContentSummaryForm),
                provides=IInnerSubForm)
class SharedContentDublinCoreSummary(InnerAdminDisplayForm):
    """Shared content DublinCore summary"""

    legend = _("Identity card")
    weight = 1
    css_class = 'form-tight'

    @property
    def fields(self):
        fields = field.Fields(IWfSharedContent).select('title') + \
                 field.Fields(ISequentialIdInfo).select('public_oid')
        return fields


@adapter_config(name='workflow-waiting-state',
                context=(IWfSharedContent, IPyAMSLayer, SharedContentSummaryForm),
                provides=IInnerSubForm)
class SharedContentWorkflowWaitingState(InnerAdminAddForm):
    """Shared content workflow waiting state summary"""

    legend = _("Requested action")
    weight = 10

    mode = DISPLAY_MODE

    def __new__(cls, context, request, form):
        workflow = IWorkflow(context)
        state = IWorkflowState(context, None)
        if (state is None) or (state.state not in workflow.waiting_states):
            return None
        return InnerAdminDisplayForm.__new__(cls)

    fields = field.Fields(IWorkflowState).select('state', 'state_urgency') + \
             field.Fields(IWorkflowStateHistoryItem).select('comment')

    def updateWidgets(self, prefix=None):
        super(SharedContentWorkflowWaitingState, self).updateWidgets(prefix)
        state = IWorkflowState(self.context)
        if 'state' in self.widgets:
            translate = self.request.localizer.translate
            workflow = IWorkflow(self.context)
            self.widgets['state'].value = translate(_("{state} {date} by {principal}")).format(
                state=translate(workflow.get_state_label(state.state)),
                date=format_datetime(state.state_date),
                principal=get_principal(self.request, state.state_principal).title
            )
        if 'state_urgency' in self.widgets:
            converter = IDataConverter(self.widgets['state_urgency'])
            self.widgets['state_urgency'].value = converter.toWidgetValue(state.state_urgency)
        if 'comment' in self.widgets:
            history_item = state.history[-1]
            self.widgets['comment'].label = _("Associated comment")
            self.widgets['comment'].value = history_item.comment


@adapter_config(name='workflow-publication-state',
                context=(IWfSharedContent, IPyAMSLayer, SharedContentSummaryForm),
                provides=IInnerSubForm)
class SharedContentWorkflowPublicationState(InnerAdminDisplayForm):
    """Shared content workflow publication state summary"""

    legend = _("Publication and retire dates")
    weight = 20

    def __new__(cls, context, request, form):
        info = IWorkflowPublicationInfo(context, None)
        if (info is None) or not info.publication_effective_date:
            return None
        return InnerAdminDisplayForm.__new__(cls)

    fields = field.Fields(IWorkflowPublicationInfo).select(
        'publication_effective_date', 'push_end_date',
        'publication_expiration_date', 'displayed_publication_date')
    
    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        info = IWorkflowPublicationInfo(self.context)
        for name in ('publication_effective_date', 'push_end_date', 'publication_expiration_date'):
            widget = self.widgets.get(name)
            if widget is not None:
                widget.value = format_datetime(getattr(info, name))


@adapter_config(name='workflow-version-summary',
                context=(IWfSharedContent, IPyAMSLayer, SharedContentSummaryForm),
                provides=IInnerSubForm)
class SharedContentWorkflowVersionSummary(InnerAdminDisplayForm):
    """Shared content workflow version summary"""

    legend = _("Current version")
    weight = 30

    def __new__(cls, context, request, form):
        state = IWorkflowState(context, None)
        if state is None:
            return None
        return InnerAdminDisplayForm.__new__(cls)

    fields = field.Fields(IWorkflowState).select('version_id', 'state') + \
             field.Fields(IWfSharedContent).select('creation_label') + \
             field.Fields(IWfSharedContentRoles).select('owner') + \
             field.Fields(IWfSharedContent).select('last_update_label') + \
             field.Fields(IWfSharedContent).select('modifiers')

    def updateWidgets(self, prefix=None):
        super(SharedContentWorkflowVersionSummary, self).updateWidgets(prefix)
        if 'version_id' in self.widgets:
            self.widgets['version_id'].label = _("Version")
        if 'state' in self.widgets:
            translate = self.request.localizer.translate
            workflow = IWorkflow(self.context)
            workflow_state = IWorkflowState(self.context)
            history_item = workflow_state.history[-1]
            self.widgets['state'].value = translate(_("{state} since {date}, by {principal}")).format(
                state=translate(workflow.get_state_label(workflow_state.state)),
                date=format_datetime(tztime(history_item.date)),
                principal=get_principal(self.request, history_item.principal).title)


@adapter_config(name='workflow-history-summary',
                context=(IWfSharedContent, IPyAMSLayer, SharedContentSummaryForm),
                provides=IInnerSubForm)
class SharedContentWorkflowHistorySummary(InnerAdminDisplayForm):
    """Shared content workflow history summary"""

    legend = _("Content history")

    def __new__(cls, context, request, form):
        info = IWorkflowPublicationInfo(context, None)
        if info is None:
            return None
        return InnerAdminDisplayForm.__new__(cls)

    @property
    def fields(self):
        fields = field.Fields(IWorkflowPublicationInfo).select('first_publication_date')
        state = IWorkflowState(self.context)
        if state.version_id > 1:
            fields += field.Fields(IWorkflowPublicationInfo).select('first_content_publication_date')
        fields += field.Fields(IWfSharedContent).select('first_owner')
        return fields

    weight = 40

    def updateWidgets(self, prefix=None):
        super(SharedContentWorkflowHistorySummary, self).updateWidgets(prefix)
        info = IWorkflowPublicationInfo(self.context, None)
        if 'first_publication_date' in self.widgets:
            self.widgets['first_publication_date'].value = \
                format_datetime(info.first_publication_date)
        if 'first_content_publication_date' in self.widgets:
            self.widgets['first_content_publication_date'].value = \
                format_datetime(info.first_content_publication_date)
