#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.exceptions import NotFound
from pyramid.url import resource_url
from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.interfaces import HIDDEN_MODE
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import implementer
from zope.schema import getFieldNamesInOrder

from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.shared.common.interfaces import IBaseSharedTool, IContributorRestrictions, \
    IManagerRestrictions
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.interfaces import IPrincipalInfo, ISecurityManager
from pyams_security.utility import get_principal
from pyams_security.zmi.interfaces import IObjectSecurityMenu
from pyams_skin.container import ContainerView
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import BaseTable, I18nColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.property import cached_property
from pyams_utils.registry import get_utility
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView


__docformat__ = 'restructuredtext'


from pyams_content import _


#
# Contributor restrictions views
#

@viewlet_config(name='contributors-restrictions.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IObjectSecurityMenu, permission=MANAGE_TOOL_PERMISSION, weight=910)
class SharedToolContributorsRestrictionsMenu(MenuItem):
    """Shared tool contributors restrictions menu"""

    label = _("Contributors restrictions")
    icon_class = 'fa-lock'
    url = '#contributors-restrictions.html'


class SharedToolContributorsRestrictionsTable(BaseTable):
    """Shared tool contributors restrictions table"""

    id = 'security_contributors_restrictions'
    title = _("Content contributors restrictions")


@adapter_config(context=(IBaseSharedTool, IAdminLayer, SharedToolContributorsRestrictionsTable),
                provides=IValues)
class SharedToolContributorsRestrictionsValuesAdapter(ContextRequestViewAdapter):
    """Shared tool contributor restrictions values adapter"""

    @property
    def values(self):
        contributor = get_utility(ISecurityManager)
        return sorted([
            contributor.get_principal(principal_id)
            for principal_id in self.context.contributors
        ], key=lambda x: x.title)


@adapter_config(context=(IPrincipalInfo, IAdminLayer, SharedToolContributorsRestrictionsTable),
                provides=ITableElementEditor)
class ContributorInfoElementEditor(ContextRequestViewAdapter):
    """Contributor info element editor"""

    view_name = 'contributor-restrictions.html'

    @property
    def url(self):
        return resource_url(self.view.context, self.request, self.view_name,
                            query={'principal_id': self.context.id})

    modal_target = True


@adapter_config(name='name',
                context=(IBaseSharedTool, IAdminLayer, SharedToolContributorsRestrictionsTable),
                provides=IColumn)
class SharedToolContributorsRestrictionsNameColumn(I18nColumn, GetAttrColumn):
    """Shared tool contributor restrictions name column"""

    _header = _("Contributor name")
    attrName = 'title'
    weight = 10


@adapter_config(name='checks',
                context=(IBaseSharedTool, IAdminLayer, SharedToolContributorsRestrictionsTable),
                provides=IColumn)
class SharedToolContributorsRestrictionsChecksColumn(I18nColumn, GetAttrColumn):
    """Shared tool contributor enabled publication checks column"""

    _header = _("Activated publication checks?")
    weight = 20
    cssClasses = {'td': 'center'}

    def getValue(self, obj):
        restrictions = IContributorRestrictions(self.context).get_restrictions(obj, create_if_none=True)
        if not restrictions.publication_checks:
            return '--'
        return '<i class="fa fa-fw fa-check"></i>'


@adapter_config(name='owners',
                context=(IBaseSharedTool, IAdminLayer, SharedToolContributorsRestrictionsTable),
                provides=IColumn)
class SharedToolContributorRestrictionsOwnersColumn(I18nColumn, GetAttrColumn):
    """Shared tool contributor restrictions owners column"""

    _header = _("Substitute for")
    weight = 25

    def getValue(self, obj):
        restrictions = IContributorRestrictions(self.context).get_restrictions(obj, create_if_none=True)
        if not restrictions.owners:
            return '--'
        return ', '.join(sorted(
            get_principal(self.request, principal_id).title
            for principal_id in restrictions.owners
        ))


@pagelet_config(name='contributors-restrictions.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@implementer(IInnerPage)
class SharedToolContributorsRestrictionsView(AdminView, ContainerView):
    """Shared tool contributors restrictions view"""

    table_class = SharedToolContributorsRestrictionsTable


@adapter_config(context=(IBaseSharedTool, IAdminLayer, SharedToolContributorsRestrictionsView),
                provides=IPageHeader)
class SharedToolContributorsRestrictionsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool contributors restrictions header adapter"""

    back_url = 'admin#protected-object-roles.html'
    back_target = None

    icon_class = 'fa fa-fw fa-lock'


#
# Contributor restrictions edit form
#

@pagelet_config(name='contributor-restrictions.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='contributor-restrictions.json',
             context=IBaseSharedTool, layer=IPyAMSLayer)
class SharedToolContributorRestrictionsEditForm(AdminDialogEditForm):
    """Shared tool contributor restrictions edit form"""

    prefix = 'tool_restrictions.'

    icon_css_class = 'fa fa-fw fa-lock'

    edit_permission = MANAGE_TOOL_PERMISSION

    @property
    def legend(self):
        return self.request.localizer.translate(
            _("Edit contributor restrictions for « {0} »")).format(self.principal.title)

    @cached_property
    def interface(self):
        restrictions = self.getContent()
        return restrictions.restriction_interface

    @property
    def fields(self):
        fields = field.Fields(self.interface)
        fields['publication_checks'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    @cached_property
    def principal_id(self):
        principal_id = self.request.params.get('principal_id') or \
                       self.request.params.get('{0}widgets.principal_id'.format(self.prefix))
        if not principal_id:
            raise NotFound
        return principal_id

    @cached_property
    def principal(self):
        manager = get_utility(ISecurityManager)
        return manager.get_principal(self.principal_id)

    def getContent(self):
        contributor_restrictions = IContributorRestrictions(self.context)
        return contributor_restrictions.get_restrictions(self.principal_id, create_if_none=True)

    def updateWidgets(self, prefix=None):
        super(SharedToolContributorRestrictionsEditForm, self).updateWidgets(prefix)
        self.widgets['principal_id'].value = self.principal
        self.widgets['principal_id'].mode = HIDDEN_MODE

    def get_ajax_output(self, changes):
        if changes:
            return {'status': 'reload'}
        else:
            return super(self.__class__, self).get_ajax_output(changes)


#
# Manager restrictions views
#

@viewlet_config(name='managers-restrictions.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IObjectSecurityMenu, permission=MANAGE_TOOL_PERMISSION, weight=920)
class SharedToolManagersRestrictionsMenu(MenuItem):
    """Shared tool managers restrictions menu"""

    label = _("Managers restrictions")
    icon_class = 'fa-lock'
    url = '#managers-restrictions.html'


class SharedToolManagersRestrictionsTable(BaseTable):
    """Shared tool manager restrictions table"""

    id = 'security_managers_restrictions'
    title = _("Content managers restrictions")


@adapter_config(context=(IBaseSharedTool, IAdminLayer, SharedToolManagersRestrictionsTable),
                provides=IValues)
class SharedToolManagersRestrictionsValuesAdapter(ContextRequestViewAdapter):
    """Shared tool manager restrictions values adapter"""

    @property
    def values(self):
        manager = get_utility(ISecurityManager)
        return sorted([
            manager.get_principal(principal_id)
            for principal_id in self.context.managers
        ], key=lambda x: x.title)


@adapter_config(context=(IPrincipalInfo, IAdminLayer, SharedToolManagersRestrictionsTable),
                provides=ITableElementEditor)
class ManagerInfoElementEditor(ContextRequestViewAdapter):
    """Principal info element editor"""

    view_name = 'manager-restrictions.html'

    @property
    def url(self):
        return resource_url(self.view.context, self.request, self.view_name,
                            query={'principal_id': self.context.id})

    modal_target = True


@adapter_config(name='name',
                context=(IBaseSharedTool, IAdminLayer, SharedToolManagersRestrictionsTable),
                provides=IColumn)
class SharedToolManagerRestrictionsNameColumn(I18nColumn, GetAttrColumn):
    """Shared tool manager restrictions name column"""

    _header = _("Manager name")
    attrName = 'title'
    weight = 10


@adapter_config(name='checks',
                context=(IBaseSharedTool, IAdminLayer, SharedToolManagersRestrictionsTable),
                provides=IColumn)
class SharedToolManagerRestrictionsChecksColumn(I18nColumn, GetAttrColumn):
    """Shared tool manager enabled publication checks column"""

    _header = _("Activated publication checks?")
    weight = 40
    cssClasses = {'td': 'center'}

    def getValue(self, obj):
        restrictions = IManagerRestrictions(self.context).get_restrictions(obj,
                                                                           create_if_none=True)
        if restrictions.publication_checks:
            return '<i class="fa fa-fw fa-check"></i>'
        else:
            return '--'


@adapter_config(name='restricted',
                context=(IBaseSharedTool, IAdminLayer, SharedToolManagersRestrictionsTable),
                provides=IColumn)
class SharedToolManagerRestrictionsEnabledColumn(I18nColumn, GetAttrColumn):
    """Shared tool manager enabled restrictions column"""

    _header = _("Restricted")
    weight = 45
    cssClasses = {'td': 'center'}

    def getValue(self, obj):
        restrictions = IManagerRestrictions(self.context).get_restrictions(obj,
                                                                           create_if_none=True)
        if restrictions.restricted_contents:
            return '<i class="fa fa-fw fa-check"></i>'
        else:
            return '--'


@adapter_config(name='owners',
                context=(IBaseSharedTool, IAdminLayer, SharedToolManagersRestrictionsTable),
                provides=IColumn)
class SharedToolManagerRestrictionsOwnersColumn(I18nColumn, GetAttrColumn):
    """Shared tool manager owners restrictions column"""

    _header = _("Owners")
    weight = 50
    cssClasses = {'td': 'center'}

    def getValue(self, obj):
        restrictions = IManagerRestrictions(self.context).get_restrictions(obj)
        if (restrictions is not None) and restrictions.restricted_contents and restrictions.owners:
            return '<i class="fa fa-fw fa-check"></i>'
        else:
            return '--'


@pagelet_config(name='managers-restrictions.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@implementer(IInnerPage)
class SharedToolManagersRestrictionsView(AdminView, ContainerView):
    """Shared tool managers restrictions view"""

    table_class = SharedToolManagersRestrictionsTable


@adapter_config(context=(IBaseSharedTool, IAdminLayer, SharedToolManagersRestrictionsView),
                provides=IPageHeader)
class SharedToolManagersRestrictionsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool managers restrictions header adapter"""

    back_url = 'admin#protected-object-roles.html'
    back_target = None

    icon_class = 'fa fa-fw fa-lock'


#
# Manager restrictions edit form
#

@pagelet_config(name='manager-restrictions.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='manager-restrictions.json',
             context=IBaseSharedTool, layer=IPyAMSLayer)
class SharedToolManagerRestrictionsEditForm(AdminDialogEditForm):
    """Shared tool manager restrictions edit form"""

    prefix = 'tool_restrictions.'

    icon_css_class = 'fa fa-fw fa-lock'

    edit_permission = MANAGE_TOOL_PERMISSION

    @property
    def legend(self):
        return self.request.localizer.translate(
            _("Edit manager restrictions for « {0} »")).format(self.principal.title)

    @cached_property
    def interface(self):
        restrictions = self.getContent()
        return restrictions.restriction_interface

    @property
    def fields(self):
        fields = field.Fields(self.interface)
        fields['publication_checks'].widgetFactory = SingleCheckBoxFieldWidget
        fields['restricted_contents'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    @cached_property
    def principal_id(self):
        principal_id = self.request.params.get('principal_id') or \
                       self.request.params.get('{0}widgets.principal_id'.format(self.prefix))
        if not principal_id:
            raise NotFound
        return principal_id

    @cached_property
    def principal(self):
        manager = get_utility(ISecurityManager)
        return manager.get_principal(self.principal_id)

    def getContent(self):
        manager_restrictions = IManagerRestrictions(self.context)
        return manager_restrictions.get_restrictions(self.principal_id, create_if_none=True)

    def updateWidgets(self, prefix=None):
        super(SharedToolManagerRestrictionsEditForm, self).updateWidgets(prefix)
        self.widgets['principal_id'].value = self.principal
        self.widgets['principal_id'].mode = HIDDEN_MODE

    def updateGroups(self):
        names = getFieldNamesInOrder(self.interface)
        names.remove('publication_checks')
        group = NamedWidgetsGroup(self, 'publication_checks', self.widgets,
                                  ('publication_checks',),
                                  legend=_("Publication workflow"),
                                  css_class='inner')
        group.label_css_class = 'control-label col-md-2'
        group.input_css_class = 'col-md-10'
        self.add_group(group)
        self.add_group(NamedWidgetsGroup(self, 'restricted_access', self.widgets, names,
                                         legend=_("Apply contents restrictions"),
                                         css_class='inner',
                                         help=_("You can specify which contents this manager "
                                                "will be able to manage. If you specify several "
                                                "criteria, the manager will be able to manage "
                                                "contents for which at least one criteria is "
                                                "matching."),
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=self.interface['restricted_contents']))
        super(SharedToolManagerRestrictionsEditForm, self).updateGroups()

    def get_ajax_output(self, changes):
        if changes:
            return {'status': 'reload'}
        return super(self.__class__, self).get_ajax_output(changes)
