#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from z3c.form import field
from z3c.form.interfaces import IDataExtractedEvent
from zope.container.interfaces import IOrderedContainer
from zope.interface import Interface, Invalid
from zope.location.interfaces import ILocation

from pyams_content.interfaces import MANAGE_SITE_PERMISSION
from pyams_content.shared.common.interfaces import ISharedSite
from pyams_content.shared.site.interfaces import IBaseSiteItem
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import ajax_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IContextActions, ITableItemColumnActionsMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm

from pyams_content import _


#
# Site container rename view
#

@viewlet_config(name='rename-item.menu', context=ISharedSite, layer=IPyAMSLayer,
                view=ISiteTreeTable, manager=ITableItemColumnActionsMenu,
                permission=MANAGE_SITE_PERMISSION, weight=900)
@viewlet_config(name='rename-item.menu', context=ISharedSite, layer=IPyAMSLayer,
                view=Interface, manager=IContextActions,
                permission=MANAGE_SITE_PERMISSION, weight=900)
@viewlet_config(name='rename-item.menu', context=IBaseSiteItem, layer=IPyAMSLayer,
                view=ISiteTreeTable, manager=ITableItemColumnActionsMenu,
                permission=MANAGE_SITE_PERMISSION, weight=900)
@viewlet_config(name='rename-item.menu', context=IBaseSiteItem, layer=IPyAMSLayer,
                view=Interface, manager=IContextActions,
                permission=MANAGE_SITE_PERMISSION, weight=900)
class SiteContainerRenameMenu(ToolbarMenuItem):
    """Site container rename menu"""

    label = _("Change URL...")
    label_css_class = 'fa fa-fw fa-edit'
    url = 'rename-item.html'
    modal_target = True


@pagelet_config(name='rename-item.html', context=ISharedSite, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='rename-item.json', context=ISharedSite, layer=IPyAMSLayer)
@pagelet_config(name='rename-item.html', context=IBaseSiteItem, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='rename-item.json', context=IBaseSiteItem, layer=IPyAMSLayer)
class SiteContainerRenameForm(AdminDialogEditForm):
    """Site container rename form"""

    prefix = 'rename_item.'

    legend = _("Change item URL")

    fields = field.Fields(ILocation).select('__name__')
    edit_permission = MANAGE_SITE_PERMISSION

    def updateWidgets(self, prefix=None):
        super(SiteContainerRenameForm, self).updateWidgets(prefix)
        self.widgets['__name__'].label = _("Item URL part")
        self.widgets['__name__'].description = _("URL part used to access this content")

    def update_content(self, content, data):
        data = data.get(self, data)
        old_name = content.__name__
        new_name = data['__name__'] = translate_string(data['__name__'], spaces='-', keep_chars='-')
        changes = super(SiteContainerRenameForm, self).update_content(content, data)
        if changes:
            # revert rename to adjust container properties
            content.__name__ = old_name
            parent = content.__parent__
            # memoize keys order before renaming object
            order = list(parent.keys())
            parent[new_name] = content
            del parent[old_name]
            # restore keys order
            if IOrderedContainer.providedBy(parent):
                order[order.index(old_name)] = new_name
                parent.updateOrder(order)
        return changes

    def get_ajax_output(self, changes):
        if changes:
            return {
                'status': 'redirect',
                'location': absolute_url(self.context, self.request, 'admin##'),
                'smallbox': {
                    'status': 'success',
                    'message': self.request.localizer.translate(self.successMessage)
                }
            }
        else:
            return super(SiteContainerRenameForm, self).get_ajax_output(changes)


@subscriber(IDataExtractedEvent, form_selector=SiteContainerRenameForm)
def handle_rename_form_data_extraction(event):
    """Handle rename form data extraction"""
    name = event.data.get('__name__')
    if not name:
        event.form.widgets.errors += (Invalid(_("You must provide an URL for this item!")),)
    target = event.form.context.__parent__.get(name)
    if (target is not None) and (target is not event.form.context):
        event.form.widgets.errors += (Invalid(_("Selected URL already exists!")),)
