# -*- coding: utf-8 -*-
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from datetime import datetime
from uuid import uuid4

from pyramid.location import lineage
from z3c.form import button, field
from zope.copy import copy
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface
from zope.lifecycleevent import ObjectCreatedEvent
from zope.location import locate

from pyams_content.features.review.interfaces import IReviewComments
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION, \
    MANAGE_SITE_ROOT_PERMISSION, PUBLISH_CONTENT_PERMISSION
from pyams_content.shared.common import IDashboardTypeColumnValue
from pyams_content.shared.common.interfaces import IBaseSharedTool, IContributorRestrictions, \
    IManagerRestrictions, ISharedContent, IWfSharedContent
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_content.shared.common.interfaces.zmi import IDashboardTable, ISiteRootDashboardTable
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IFormContextPermissionChecker, IWidgetsPrefixViewletsManager
from pyams_form.schema import CloseButton
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_i18n.widget import I18nSEOTextLineFieldWidget
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.interfaces import IContentTitle
from pyams_skin.interfaces.container import ITable, ITableElementEditor
from pyams_skin.interfaces.viewlet import IBreadcrumbItem, IContextActions, IMenuHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import DefaultElementEditorAdapter
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.toolbar import ToolbarMenuDivider, ToolbarMenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, ContextRequestViewAdapter, \
    adapter_config
from pyams_utils.interfaces import FORBIDDEN_PERMISSION
from pyams_utils.registry import get_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, generate_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowCommentInfo, IWorkflowInfo, \
    IWorkflowPublicationInfo, IWorkflowState, IWorkflowVersions
from pyams_workflow.versions import WorkflowHistoryItem
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.interfaces.menu import ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


class SharedContentAddForm(AdminDialogAddForm):
    """Shared content add form"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    icon_css_class = 'fa fa-fw fa-plus'
    fields = field.Fields(IWfSharedContent).select('title', 'notepad')
    fields['title'].widgetFactory = I18nSEOTextLineFieldWidget

    ajax_handler = 'add-shared-content.json'
    edit_permission = CREATE_CONTENT_PERMISSION

    __uuid = None

    def updateWidgets(self, prefix=None):
        super(SharedContentAddForm, self).updateWidgets(prefix)
        if 'title' in self.widgets:
            self.widgets['title'].description = _("This title can be modified afterwards")

    def create(self, data):
        return self.context.shared_content_factory.content_class()

    def update_content(self, content, data):
        changes = super(SharedContentAddForm, self).update_content(content, data)
        # initialize content fields
        lang = get_utility(INegotiator).server_language
        content.creator = self.request.principal.id
        content.owner = self.request.principal.id
        content.short_name = content.title.copy()
        content.content_url = generate_url(content.title.get(lang, ''))
        # init content languages
        languages = II18nManager(self.context).languages
        if languages:
            II18nManager(content).languages = languages.copy()
        return changes

    def add(self, wf_content):
        content = self.context.shared_content_factory()
        self.request.registry.notify(ObjectCreatedEvent(content))
        self.__uuid = uuid = str(uuid4())
        self.context[uuid] = content
        IWorkflowVersions(content).add_version(wf_content, None)
        IWorkflowInfo(wf_content).fire_transition('init', comment=wf_content.notepad)

    def nextURL(self):
        return absolute_url(self.context, self.request,
                            '{0}/++versions++/1/admin#properties.html'.format(self.__uuid))


class SharedContentAJAXAddForm(AJAXAddForm):
    """Shared event add form, JSON renderer"""

    def get_ajax_output(self, changes):
        return {
            'status': 'redirect',
            'location': self.nextURL()
        }


@viewlet_config(name='wf-create-message',
                context=Interface, layer=IPyAMSLayer,
                view=SharedContentAddForm,
                manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-create-message.pt')
class SharedContentAddFormMessage(Viewlet):
    """Shared content add form info message"""


#
# Shared tools common adapters
#

@adapter_config(context=(IWfSharedContent, IAdminLayer),
                provides=IBreadcrumbItem)
class WfSharedContentBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """Shared content breadcrumb adapter"""

    @property
    def label(self):
        return II18n(self.context).query_attribute('title', request=self.request)


#
# Edit adapters and views
#

@adapter_config(context=IWfSharedContent,
                provides=IFormContextPermissionChecker)
class WfSharedContentPermissionChecker(ContextAdapter):
    """Shared content form permission checker"""

    @property
    def edit_permission(self):
        workflow = IWorkflow(self.context)
        state = IWorkflowState(self.context).state
        if state in workflow.readonly_states:
            # forbidden access to all for archived contents
            return FORBIDDEN_PERMISSION
        elif state in workflow.protected_states:
            # webmaster can update published contents
            return MANAGE_SITE_ROOT_PERMISSION
        else:
            request = check_request()
            if request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=self.context):
                # webmaster access
                return MANAGE_SITE_ROOT_PERMISSION
            principal_id = request.principal.id
            if state in workflow.manager_states:
                # restricted manager access
                if principal_id in self.context.managers:
                    return PUBLISH_CONTENT_PERMISSION
                for parent in lineage(self.context):
                    manager_restrictions = IManagerRestrictions(parent, None)
                    if manager_restrictions is not None:
                        user_restrictions = manager_restrictions.get_restrictions(principal_id)
                        if user_restrictions:
                            if user_restrictions.check_access(self.context,
                                                              permission=PUBLISH_CONTENT_PERMISSION,
                                                              request=request):
                                return PUBLISH_CONTENT_PERMISSION
                            else:
                                return FORBIDDEN_PERMISSION
            else:
                # access is granted  to content's owner and designated contributors or managers
                if principal_id in self.context.owner | \
                                   self.context.contributors | \
                                   self.context.managers:
                    return MANAGE_CONTENT_PERMISSION
                # check if current principal can manage owner's contents
                for parent in lineage(self.context):
                    contrib_restrictions = IContributorRestrictions(parent, None)
                    if contrib_restrictions is not None:
                        user_restrictions = contrib_restrictions.get_restrictions(principal_id)
                        if user_restrictions:
                            if user_restrictions.check_access(self.context,
                                                              permission=MANAGE_CONTENT_PERMISSION,
                                                              request=request):
                                return MANAGE_CONTENT_PERMISSION
                restrictions = IContributorRestrictions(self.context).get_restrictions(principal_id)
                if restrictions and restrictions.check_access(self.context,
                                                              permission=MANAGE_CONTENT_PERMISSION,
                                                              request=request):
                    return MANAGE_CONTENT_PERMISSION
                # check if current principal can manage content's due to manager restrictions
                for parent in lineage(self.context):
                    manager_restrictions = IManagerRestrictions(parent, None)
                    if manager_restrictions is not None:
                        user_restrictions = manager_restrictions.get_restrictions(principal_id)
                        if user_restrictions:
                            if user_restrictions.check_access(self.context,
                                                              permission=MANAGE_CONTENT_PERMISSION,
                                                              request=request):
                                return MANAGE_CONTENT_PERMISSION
                restrictions = IManagerRestrictions(self.context).get_restrictions(principal_id)
                if restrictions and restrictions.check_access(self.context,
                                                              permission=MANAGE_CONTENT_PERMISSION,
                                                              request=request):
                    return MANAGE_CONTENT_PERMISSION
        return FORBIDDEN_PERMISSION


class WfSharedContentPermissionMixin(object):
    """Shared content permission checker"""

    @property
    def permission(self):
        content = get_parent(self.context, IWfSharedContent)
        if content is not None:
            return IFormContextPermissionChecker(content).edit_permission


@adapter_config(context=(IWfSharedContent, ISiteManagementMenu),
                provides=IMenuHeader)
class WfSharedContentSiteManagementMenuHeader(ContextRequestAdapter):
    """Shared content site management menu header adapter"""

    header = ' '


@adapter_config(context=(IWfSharedContent, IPyAMSLayer, Interface),
                provides=IContentTitle)
class WfSharedContentTitleAdapter(ContextRequestViewAdapter):
    """Shared content title adapter"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)


class WfSharedContentHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared content header adapter"""

    @property
    def back_url(self):
        shared_tool = get_parent(self.context, IBaseSharedTool)
        return absolute_url(shared_tool, self.request, 'admin#dashboard.html')

    back_target = None
    icon_class = 'fa fa-fw fa-edit'


@adapter_config(context=(IWfSharedContent, IPyAMSLayer, ITable),
                provides=ITableElementEditor)
class WfSharedContentElementEditor(DefaultElementEditorAdapter):
    """Shared content element editor"""

    view_name = 'admin'
    modal_target = False


#
# Duplication menus and views
#

@viewlet_config(name='duplication.divider',
                context=IWfSharedContent, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, weight=49,
                permission=CREATE_CONTENT_PERMISSION)
class WfSharedContentDuplicationMenuDivider(ToolbarMenuDivider):
    """Shared content duplication menu divider"""


@viewlet_config(name='duplication.menu',
                context=IWfSharedContent, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, weight=50,
                permission=CREATE_CONTENT_PERMISSION)
class WfSharedContentDuplicateMenu(ToolbarMenuItem):
    """Shared content duplication menu item"""

    label = _("Duplicate content...")
    label_css_class = 'fa fa-fw fa-files-o'

    url = 'duplicate.html'
    modal_target = True


class ISharedContentDuplicateButtons(Interface):
    """Shared content duplication form buttons"""

    close = CloseButton(name='close',
                        title=_("Cancel"))
    duplicate = button.Button(name='duplicate',
                              title=_("Duplicate this content"))


@pagelet_config(name='duplicate.html',
                context=IWfSharedContent, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='duplicate.json',
             context=IWfSharedContent, layer=IPyAMSLayer,
             base=AJAXAddForm)
class WfSharedContentDuplicateForm(AdminDialogAddForm):
    """Shared content duplicate form"""

    legend = _("Duplicate content")
    icon_css_class = 'fa fa-fw fa-files-o'

    fields = field.Fields(IWorkflowCommentInfo)
    buttons = button.Buttons(ISharedContentDuplicateButtons)

    edit_permission = CREATE_CONTENT_PERMISSION

    def updateActions(self):
        super(WfSharedContentDuplicateForm, self).updateActions()
        if 'duplicate' in self.actions:
            self.actions['duplicate'].addClass('btn-primary')

    def createAndAdd(self, data):
        data = data.get(self, data)
        registry = self.request.registry
        # initialize new content
        content = get_parent(self.context, ISharedContent)
        new_content = content.__class__()
        registry.notify(ObjectCreatedEvent(new_content))
        container = get_parent(content, IBaseSharedTool)
        container[str(uuid4())] = new_content
        # initialize new version
        new_version = copy(self.context)
        registry.notify(ObjectCreatedEvent(new_version))
        locate(new_version, self.context.__parent__)  # locate new version for traversing to work...
        new_version.creator = self.request.principal.id
        new_version.owner = self.request.principal.id
        new_version.modifiers = set()
        # check comments
        comments = IReviewComments(new_version, None)
        if comments is not None:
            comments.clear()
        # store new version
        translate = self.request.localizer.translate
        workflow = get_utility(IWorkflow, name=new_content.workflow_name)
        IWorkflowVersions(new_content).add_version(new_version, workflow.initial_state)
        # reset publication info
        pub_info = IWorkflowPublicationInfo(new_version, None)
        if pub_info is not None:
            pub_info.reset(complete=True)
        # update new version history
        source_state = IWorkflowState(self.context)
        state = IWorkflowState(new_version)
        state.history.clear()
        history = WorkflowHistoryItem(date=datetime.utcnow(),
                                      principal=self.request.principal.id,
                                      comment=translate(
                                          _("Clone created from version {source} of {oid} "
                                            "(in « {state} » state)")).format(
                                          source=source_state.version_id,
                                          oid=ISequentialIdInfo(self.context).get_short_oid(),
                                          state=translate(IWorkflow(self.context).get_state_label(
                                              source_state.state)))
                                      )
        state.history.append(history)
        history = WorkflowHistoryItem(date=datetime.utcnow(),
                                      target_state=workflow.initial_state,
                                      principal=self.request.principal.id,
                                      comment=data.get('comment'))
        state.history.append(history)
        return new_version

    def get_ajax_output(self, changes):
        return {
            'status': 'redirect',
            'location': absolute_url(changes, self.request, 'admin')
        }


@viewlet_config(name='wf-duplicate-message',
                context=IWfSharedContent, layer=IPyAMSLayer, view=WfSharedContentDuplicateForm,
                manager=IWidgetsPrefixViewletsManager, weight=20)
@template_config(template='templates/wf-duplicate-message.pt')
class WfSharedContentDuplicateFormMessage(Viewlet):
    """Shared content add form info message"""


#
# Custom columns mixins
#

@adapter_config(context=(IWfSharedContent, IPyAMSLayer, IDashboardTable),
                provides=IDashboardTypeColumnValue)
def typed_shared_content_tool_dashboard_type_adapter(context, request, view):
    """Typed shared content dashboard type column adapter"""
    content = IWfTypedSharedContent(context, None)
    if content is not None:
        data_type = content.get_data_type()
        if data_type is not None:
            return II18n(data_type).query_attributes_in_order(('backoffice_label', 'label'),
                                                              request=request)


@adapter_config(context=(IWfSharedContent, IPyAMSLayer, ISiteRootDashboardTable),
                provides=IDashboardTypeColumnValue)
def typed_shared_content_root_dashboard_type_adapter(context, request, view):
    """Typed shared content root dashboard type column adapter"""
    tool = get_parent(context, IBaseSharedTool)
    result = II18n(tool).query_attribute('backoffice_label', request=request)
    if not result:
        result = request.localizer.translate(context.content_name)
    content = IWfTypedSharedContent(context, None)
    if content is not None:
        data_type = content.get_data_type()
        if data_type is not None:
            result = '{} ({})'.format(
                result,
                II18n(data_type).query_attributes_in_order(('backoffice_label', 'label'),
                                                           request=request))
    return result or '--'


class WfModifiedContentColumnMixin(object):
    """Shared content modified column mixin"""

    def renderCell(self, item):
        value = self.getValue(item)
        content = get_parent(item, IWfSharedContent)
        if content is not None:
            if IWorkflowState(content).version_id > 1:
                item_dc = IZopeDublinCore(item)
                if item_dc.modified and (item_dc.modified > IZopeDublinCore(content).created):
                    translate = self.request.localizer.translate
                    value += '<i class="fa fa-fw fa-circle txt-color-orange hint align-middle ' \
                             'padding-left-5" title="{0}" data-ams-hint-gravity="w"></i>'.format(
                        translate(_("Created or modified in this version")))
        return value
