#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from datetime import datetime, timedelta

from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq, Gt, Lt, Or
from pyramid.events import subscriber

from pyams_content.shared.common import IBaseSharedTool, ISharedContent, IWfSharedContent, IWfSharedContentCreator, \
    IWfSharedContentUpdater
from pyams_content.shared.common.interfaces import IWfSharedContentFinderParams
from pyams_content.shared.common.manager import SharedToolContentFinder
from pyams_content.workflow.basic import PUBLISHED
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.timezone import gmtime, tztime
from pyams_utils.traversing import get_parent
from pyams_workflow.interfaces import AmbiguousTransitionError, IWorkflow, IWorkflowInfo, IWorkflowPublicationInfo, \
    IWorkflowState, IWorkflowTransitionEvent


@adapter_config(name='workflow',
                required=IWfSharedContent,
                provides=IWfSharedContentCreator)
class WfSharedContentWorkflowCreator(ContextAdapter):
    """Shared content workflow creator"""

    weight = 999

    def update_content(self, params, request=None):
        """Shared content workflow updated"""
        now = gmtime(datetime.utcnow())
        publication_date = params.get('publication_date')
        pub_info = IWorkflowPublicationInfo(self.context)
        pub_info.publication_effective_date = publication_date or now
        pub_info.publication_expiration_date = params.get('expiration_date')
        state = params.get('state')
        if state == PUBLISHED:
            if request is None:
                request = check_request()
            wf_info = IWorkflowInfo(self.context)
            if (pub_info.publication_effective_date - now).total_seconds() < 60:
                wf_info.fire_transition_toward(PUBLISHED,
                                               principal=request.principal.id,
                                               check_security=False)
            else:
                workflow = IWorkflow(self.context)
                parent = workflow.get_transition_by_id('draft_to_published')
                if parent is None:
                    raise AmbiguousTransitionError("Missing transition!")
                transition = parent.user_data.get('prepared_transition')
                if transition is None:
                    raise AmbiguousTransitionError("This workflow doesn't support pre-publication!")
                wf_info.fire_transition(transition.transition_id,
                                        principal=request.principal.id,
                                        check_security=False)
                wf_info.fire_automatic()


@adapter_config(name='workflow',
                required=IWfSharedContent,
                provides=IWfSharedContentUpdater)
class WfSharedContentWorkflowUpdater(ContextAdapter):
    """Shared content workflow updater"""

    weight = -999

    def update_content(self, params, request=None):
        """Shared content workflow updater"""

        # set workflow state
        result = None
        workflow = IWorkflow(self.context)
        wf_state = IWorkflowState(self.context)
        if wf_state.state not in workflow.update_states:
            if request is None:
                request = check_request()
            result = IWorkflowInfo(self.context).fire_transition_toward(workflow.initial_state,
                                                                        principal=request.principal.id,
                                                                        check_security=False)
            if result is None:
                result = self.context
            IWorkflowInfo(result).fire_automatic()
        if result is None:
            result = self.context
        return result


@adapter_config(name='workflow-publication',
                required=IWfSharedContent,
                provides=IWfSharedContentUpdater)
class WfSharedContentWorkflowPublicationAdapter(ContextAdapter):
    """Shared content workflow publication updater"""

    weight = 999

    def update_content(self, params, request=None):

        def set_workflow_dates(version):
            """Set workflow dates on version"""
            pub_info = IWorkflowPublicationInfo(version)
            if 'publication_date' in params:
                pub_info.publication_effective_date = params['publication_date']
            if 'expiration_date' in params:
                pub_info.publication_expiration_date = params['expiration_date']
            if (state in workflow.published_states) and \
                    (pub_info.publication_effective_date is None):
                pub_info.publication_effective_date = now

        state = params.get('state')
        if not state:
            return
        now = gmtime(datetime.utcnow())
        workflow = IWorkflow(self.context)
        wf_state = IWorkflowState(self.context)
        set_workflow_dates(self.context)
        if state != wf_state.state:
            wf_info = IWorkflowInfo(self.context)
            if state in workflow.published_states:
                pub_info = IWorkflowPublicationInfo(self.context)
                if pub_info.publication_effective_date < (now + timedelta(seconds=1)):
                    wf_info.fire_transition_toward(state,
                                                   side_effect=set_workflow_dates,
                                                   principal=request.principal.id,
                                                   check_security=False)
                else:
                    transition = workflow.get_transition(wf_state.state,
                                                         '{}_to_published'.format(wf_state.state))
                    prepared_transition = transition.user_data.get('prepared_transition')
                    if prepared_transition is None:
                        raise AmbiguousTransitionError("This workflow doesn't support pre-publication!")
                    wf_info.fire_transition(prepared_transition.transition_id,
                                            principal=request.principal.id,
                                            check_security=False)
            wf_info.fire_automatic()


@adapter_config(required=IWfSharedContent,
                provides=IWorkflow)
def wf_shared_content_workflow_adapter(context):
    """Shared content workflow adapter"""
    parent = get_parent(context, IBaseSharedTool)
    return query_utility(IWorkflow, name=parent.shared_content_workflow)


@adapter_config(required=ISharedContent,
                provides=IWorkflow)
def shared_content_workflow_adapter(context):
    """Shared content workflow adapter"""
    parent = get_parent(context, IBaseSharedTool)
    return query_utility(IWorkflow, name=parent.shared_content_workflow)


@subscriber(IWorkflowTransitionEvent)
def handle_workflow_event(event):
    """Remove visible version on workflow transition"""
    content = get_parent(event.object, ISharedContent)
    if content is not None:
        del content.visible_version


@adapter_config(name='workflow',
                required=SharedToolContentFinder,
                provides=IWfSharedContentFinderParams)
class SharedToolContentFinderWorkflowParams(ContextAdapter):
    """Shared tool content finder workflow params"""

    def get_params(self, query, params, request=None, **kwargs):
        """Query params getter"""
        catalog = kwargs.get('catalog')
        if catalog is None:
            catalog = get_utility(ICatalog)
        now = tztime(datetime.utcnow())
        if 'state' in params:
            state = params['state']
            if not isinstance(state, (list, tuple, set)):
                state = state.split(',')
            query &= Any(catalog['workflow_state'], state)
            if state == PUBLISHED:
                query &= Lt(catalog['effective_date'], now)
                query &= Or(Eq(catalog['expiration_date'], None),
                            Gt(catalog['expiration_date'], now))
        if 'age_limit' in params:
            query &= Gt(catalog['content_publication_date'],
                        now - timedelta(days=params['age_limit']))
        return query
