#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form.field import Fields
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import BaseParagraph
from pyams_content.features.json import JSONBaseConverter
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.shared.common.api.json import JSONSharedContentConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_content.shared.common.interfaces.specificities import ISharedContentSpecificitiesParagraph, \
    SHARED_CONTENT_SPECIFICITIES_PARAGRAPH_RENDERERS
from pyams_content.shared.common.interfaces.types import ITypedSharedTool, IWfTypedSharedContent
from pyams_i18n.interfaces import INegotiator
from pyams_i18n.interfaces.schema import II18nField
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config

from pyams_content import _


#
# Shared content specificities exporter
#

@adapter_config(name='specificities',
                required=(IWfTypedSharedContent, IPyAMSLayer),
                provides=IJSONConverter)
class SharedContentSpecificitiesConverter(JSONBaseConverter):
    """JSON typed shared content specificities exporter"""

    is_inner = True

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        result = super().convert_content(params)
        context = self.context
        datatype = context.get_data_type()
        if datatype is not None:
            manager = get_parent(context, ITypedSharedTool)
            fields_interface = manager.shared_content_types_fields
            if fields_interface is not None:
                target = fields_interface(context, None)
                if target is not None:
                    for name, field in Fields(fields_interface).select(*datatype.field_names).items():
                        if II18nField.providedBy(field.field):
                            self.get_i18n_attribute(context, name, params.get('lang'), result)
                        else:
                            attr = field.field.bind(target).query(target)
                            if attr and II18nField.providedBy(field.field):
                                negotiator = get_utility(INegotiator)
                                attr = attr.get(negotiator.server_language)
                            if attr:
                                result[name] = attr
        return result


#
# Shared content specificities paragraph
#

SHARED_CONTENT_SPECIFICITIES_PARAGRAPH_TYPE = 'ContentSpecificities'
SHARED_CONTENT_SPECIFICITIES_PARAGRAPH_NAME = _('Content specificities')


@factory_config(provided=ISharedContentSpecificitiesParagraph)
class SharedContentSpecificitiesParagraph(BaseParagraph):
    """Shared content specificities paragraph"""

    factory_name = SHARED_CONTENT_SPECIFICITIES_PARAGRAPH_TYPE
    icon_hint = _("Content specificities")
    icon_class = 'fa-paperclip'

    renderer = FieldProperty(ISharedContentSpecificitiesParagraph['renderer'])


@vocabulary_config(name=SHARED_CONTENT_SPECIFICITIES_PARAGRAPH_RENDERERS)
class SharedContentSpecificitiesParagraphRendererVocabulary(RenderersVocabulary):
    """Shared content specificities renderers vocabulary"""

    content_interface = ISharedContentSpecificitiesParagraph
