#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.folder import Folder
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.interfaces import CONTRIBUTOR_ROLE, MANAGER_ROLE, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import CONTRIBUTOR_RESTRICTIONS_KEY, IBaseSharedTool, \
    IContributorRestrictionInfo, IContributorRestrictions, IContributorRestrictionsFactory, \
    IManagerRestrictionInfo, IManagerRestrictions, IManagerRestrictionsFactory, IRestrictionInfo, \
    IRestrictions, IWfSharedContent, MANAGER_RESTRICTIONS_KEY, \
    SHARED_TOOL_CONTRIBUTORS_VOCABULARY
from pyams_security.interfaces import IGrantedRoleEvent, IPrincipalInfo, IRevokedRoleEvent
from pyams_security.utility import get_principal
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.request import check_request, query_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'


@implementer(IRestrictionInfo)
class PrincipalRestrictionInfo(Persistent):
    """Principal restriction info"""

    principal_id = FieldProperty(IManagerRestrictionInfo['principal_id'])

    def __init__(self, principal_id):
        self.principal_id = principal_id


@implementer(IRestrictions)
class PrincipalRestrictions(ContextAdapter):
    """Shared tool restrictions"""

    restrictions_key = None
    restrictions_factory_interface = None

    def new_restrictions(self, principal):
        if IPrincipalInfo.providedBy(principal):
            principal = principal.id
        factory = self.restrictions_factory_interface(self.context)
        return factory(principal)

    def get_restrictions(self, principal, create_if_none=False):
        if IPrincipalInfo.providedBy(principal):
            principal = principal.id
        restrictions_folder = get_annotation_adapter(self.context, self.restrictions_key, Folder)
        restrictions = restrictions_folder.get(principal)
        if (restrictions is None) and create_if_none:
            restrictions = self.new_restrictions(principal)
            self.set_restrictions(principal, restrictions)
        return restrictions

    def set_restrictions(self, principal, restrictions=None):
        if IPrincipalInfo.providedBy(principal):
            principal = principal.id
        restrictions_folder = get_annotation_adapter(self.context, self.restrictions_key, Folder)
        if principal not in restrictions_folder:
            if restrictions is None:
                restrictions = self.new_restrictions(principal)
            restrictions_folder[principal] = restrictions

    def drop_restrictions(self, principal):
        restrictions_folder = get_annotation_adapter(self.context, self.restrictions_key)
        if restrictions_folder is None:
            return
        if IPrincipalInfo.providedBy(principal):
            principal = principal.id
        if principal in restrictions_folder:
            del restrictions_folder[principal]


#
# Contributor restrictions
#

@implementer(IContributorRestrictionInfo)
class SharedToolContributorRestrictionInfo(PrincipalRestrictionInfo):
    """Shared tool contributor restriction info"""

    restriction_interface = IContributorRestrictionInfo

    publication_checks = FieldProperty(IContributorRestrictionInfo['publication_checks'])
    owners = FieldProperty(IContributorRestrictionInfo['owners'])

    def check_access(self, context, permission=MANAGE_CONTENT_PERMISSION, request=None):
        if request is None:
            request = check_request()
        if not request.has_permission(permission, context=context):  # check permission
            return False
        if context.owner & (self.owners or set()):  # check if owners are matching
            return True
        return False


@adapter_config(context=IBaseSharedTool, provides=IContributorRestrictions)
class SharedToolContributorRestrictions(PrincipalRestrictions):
    """Shared tool contributor restrictions"""

    restrictions_key = CONTRIBUTOR_RESTRICTIONS_KEY
    restrictions_factory_interface = IContributorRestrictionsFactory


@subscriber(IGrantedRoleEvent, role_selector=CONTRIBUTOR_ROLE)
def handle_added_contributor_role(event):
    """Handle added contributor role"""
    shared_tool = event.object.__parent__
    contributor_restrictions = IContributorRestrictions(shared_tool, None)
    if contributor_restrictions:
        contributor_restrictions.set_restrictions(event.principal_id)


@subscriber(IRevokedRoleEvent, role_selector=CONTRIBUTOR_ROLE)
def handle_revoked_contributor_role(event):
    """Handle revoked contributor role"""
    restrictions = IContributorRestrictions(event.object.__parent__, None)
    if restrictions:
        restrictions.drop_restrictions(event.principal_id)


@adapter_config(context=IWfSharedContent, provides=IContributorRestrictions)
def shared_content_contributor_restrictions(context):
    """Shared tool contributor restrictions"""
    tool = get_parent(context, IBaseSharedTool)
    if tool is not None:
        return IContributorRestrictions(tool)


@adapter_config(context=IBaseSharedTool, provides=IContributorRestrictionsFactory)
def shared_tool_contributor_restrictions_factory(context):
    """Default shared tool contributor restrictions factory"""
    return SharedToolContributorRestrictionInfo


@vocabulary_config(name=SHARED_TOOL_CONTRIBUTORS_VOCABULARY)
class SharedContentContributorsVocabulary(SimpleVocabulary):
    """Shared content contributors vocabulary"""

    def __init__(self, context=None):
        terms = []
        shared_tool = get_parent(context, IBaseSharedTool)
        if shared_tool is not None:
            request = query_request()
            terms = sorted([
                SimpleTerm(principal_id, title=get_principal(request, principal_id).title)
                for principal_id in shared_tool.contributors
            ], key=lambda x: x.title)
        super().__init__(terms)


#
# Manager restrictions
#

@implementer(IManagerRestrictionInfo)
class SharedToolManagerRestrictionInfo(PrincipalRestrictionInfo):
    """Shared tool manager restriction info"""

    restriction_interface = IManagerRestrictionInfo

    publication_checks = FieldProperty(IManagerRestrictionInfo['publication_checks'])
    restricted_contents = FieldProperty(IManagerRestrictionInfo['restricted_contents'])
    owners = FieldProperty(IManagerRestrictionInfo['owners'])

    def check_access(self, context, permission=MANAGE_CONTENT_PERMISSION, request=None):
        if request is None:
            request = check_request()
        if not request.has_permission(permission, context=context):  # check permission
            return False
        if not self.restricted_contents:  # get access if no restriction
            return True
        if context.owner & (self.owners or set()):  # check if owners are matching
            return True
        return False


@adapter_config(context=IBaseSharedTool, provides=IManagerRestrictions)
class SharedToolManagerRestrictions(PrincipalRestrictions):
    """Shared tool manager restrictions"""

    restrictions_key = MANAGER_RESTRICTIONS_KEY
    restrictions_factory_interface = IManagerRestrictionsFactory


@subscriber(IGrantedRoleEvent, role_selector=MANAGER_ROLE)
def handle_added_manager_role(event):
    """Handle added manager role"""
    shared_tool = event.object.__parent__
    manager_restrictions = IManagerRestrictions(shared_tool, None)
    if manager_restrictions:
        manager_restrictions.set_restrictions(event.principal_id)


@subscriber(IRevokedRoleEvent, role_selector=MANAGER_ROLE)
def handle_revoked_manager_role(event):
    """Handle revoked manager role"""
    restrictions = IManagerRestrictions(event.object.__parent__, None)
    if restrictions:
        restrictions.drop_restrictions(event.principal_id)


@adapter_config(context=IWfSharedContent, provides=IManagerRestrictions)
def shared_content_manager_restrictions(context):
    """Shared tool manager restrictions"""
    tool = get_parent(context, IBaseSharedTool)
    if tool is not None:
        return IManagerRestrictions(tool)


@adapter_config(context=IBaseSharedTool, provides=IManagerRestrictionsFactory)
def shared_tool_manager_restrictions_factory(context):
    """Default shared tool manager restrictions factory"""
    return SharedToolManagerRestrictionInfo
