#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from datetime import datetime, timedelta
from uuid import uuid4

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Eq, Gt
from pyramid.events import subscriber
from zope.annotation.interfaces import IAttributeAnnotatable
from zope.component.interfaces import ISite
from zope.container.folder import Folder
from zope.interface import implementer
from zope.lifecycleevent import ObjectCreatedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.location import locate
from zope.schema.fieldproperty import FieldProperty

from pyams_catalog.query import CatalogResultSet
from pyams_content.interfaces import CONTENT_PUBLICATION_DATE_ORDER, CONTRIBUTOR_ROLE, MANAGER_ROLE, PILOT_ROLE, \
    WEBMASTER_ROLE
from pyams_content.shared.common.interfaces import IBaseSharedTool, ISharedContentFactory, ISharedTool, \
    ISharedToolContainer, ISharedToolRoles, IWfSharedContentCreator, IWfSharedContentFinder, \
    IWfSharedContentFinderParams
from pyams_i18n.content import I18nManagerMixin
from pyams_portal.interfaces import DESIGNER_ROLE
from pyams_security.interfaces import IDefaultProtectionPolicy, ISecurityManager
from pyams_security.property import RolePrincipalsFieldProperty
from pyams_security.security import ProtectedObject
from pyams_utils.adapter import ContextAdapter, adapter_config, get_adapter_weight
from pyams_utils.list import boolean_iter, unique_iter
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_workflow.interfaces import IWorkflow, IWorkflowInfo, IWorkflowVersions


@implementer(ISharedToolContainer, IAttributeAnnotatable)
class SharedToolContainer(Folder):
    """Shared tools container"""

    title = FieldProperty(ISharedToolContainer['title'])
    short_name = FieldProperty(ISharedToolContainer['short_name'])


@implementer(IDefaultProtectionPolicy, IBaseSharedTool, ISharedToolRoles)
class BaseSharedTool(ProtectedObject, I18nManagerMixin):
    """Base shared tool class"""

    __roles__ = (WEBMASTER_ROLE, PILOT_ROLE, MANAGER_ROLE, DESIGNER_ROLE, CONTRIBUTOR_ROLE)

    roles_interface = ISharedToolRoles

    api_clients = RolePrincipalsFieldProperty(ISharedToolRoles['api_clients'])
    webmasters = RolePrincipalsFieldProperty(ISharedToolRoles['webmasters'])
    pilots = RolePrincipalsFieldProperty(ISharedToolRoles['pilots'])
    managers = RolePrincipalsFieldProperty(ISharedToolRoles['managers'])
    contributors = RolePrincipalsFieldProperty(ISharedToolRoles['contributors'])
    designers = RolePrincipalsFieldProperty(ISharedToolRoles['designers'])

    title = FieldProperty(IBaseSharedTool['title'])
    short_name = FieldProperty(IBaseSharedTool['short_name'])

    shared_content_menu = True
    shared_content_workflow = FieldProperty(IBaseSharedTool['shared_content_workflow'])

    label = FieldProperty(IBaseSharedTool['label'])
    navigation_label = FieldProperty(IBaseSharedTool['navigation_label'])
    facets_label = FieldProperty(IBaseSharedTool['facets_label'])
    facets_type_label = FieldProperty(IBaseSharedTool['facets_type_label'])
    backoffice_label = FieldProperty(IBaseSharedTool['backoffice_label'])


@implementer(ISharedTool)
class SharedTool(Folder, BaseSharedTool):
    """Shared tool class"""

    @property
    def shared_content_factory(self):
        return ISharedContentFactory(self, None)

    @property
    def shared_content_type(self):
        factory = self.shared_content_factory
        if factory is not None:
            return factory.content_class.content_type

    def create(self, params, request=None):
        """Create new content from given params"""
        if request is None:
            request = check_request()
        factory = self.shared_content_factory
        if factory is None:
            return
        version = factory.content_class()
        locate(version, self)
        request.registry.notify(ObjectCreatedEvent(version))
        creator = IWfSharedContentCreator(version, None)
        if creator is not None:
            creator.update_content(params, request)
        content = factory()
        request.registry.notify(ObjectCreatedEvent(content))
        uuid = str(uuid4())
        self[uuid] = content
        IWorkflowVersions(content).add_version(version, None)
        IWorkflowInfo(version).fire_transition('init')
        for name, adapter in sorted(request.registry.getAdapters((version,),
                                                                 IWfSharedContentCreator),
                                    key=get_adapter_weight):
            if not name:
                continue
            adapter.update_content(params, request)
        return version

    def find(self, params, request=None,
             sort_index=CONTENT_PUBLICATION_DATE_ORDER, reverse=True,
             limit=999):
        """Find contents matching given params"""
        finder = IWfSharedContentFinder(self, None)
        if finder is None:
            return ()
        yield from finder.get_results(params, request, sort_index, reverse, limit)


@subscriber(IObjectAddedEvent, context_selector=ISharedTool)
def handle_added_shared_tool(event):
    """Handle added shared tool"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        tool = event.object
        registry.registerUtility(tool, ISharedTool, name=tool.shared_content_type)


@adapter_config(required=IBaseSharedTool,
                provides=IWorkflow)
def shared_tool_workflow_adapter(context):
    """Shared tool workflow adapter"""
    return query_utility(IWorkflow, name=context.shared_content_workflow)


@adapter_config(required=IBaseSharedTool,
                provides=IWfSharedContentFinder)
class SharedToolContentFinder(ContextAdapter):
    """Shared tool content finder"""

    def get_results(self, params, request=None,
                    sort_index=CONTENT_PUBLICATION_DATE_ORDER, reverse=True,
                    limit=999):
        """Search results getter"""
        if request is None:
            request = check_request()
        registry = request.registry
        catalog = get_utility(ICatalog)
        query = Eq(catalog['content_type'], self.context.shared_content_type)
        for name, adapter in registry.getAdapters((self,), IWfSharedContentFinderParams):
            query = adapter.get_params(query, params, request, catalog=catalog)
        yield from unique_iter(CatalogResultSet(CatalogQuery(catalog).query(query,
                                                                            sort_index=sort_index,
                                                                            reverse=reverse,
                                                                            limit=limit or 999)))


@adapter_config(name='main',
                required=SharedToolContentFinder,
                provides=IWfSharedContentFinderParams)
class SharedToolContentFinderMainParams(ContextAdapter):
    """Shared tool content finder main params getter"""

    def get_params(self, query, params, request=None, **kwargs):
        """Search results params getter"""
        catalog = kwargs.get('catalog')
        if catalog is None:
            catalog = get_utility(ICatalog)
        if 'data_type' in params:
            query &= Eq(catalog['data_type'], params['data_type'])
        if 'owner' in params:
            owner = params['owner']
            if '@' in owner:
                sm = get_utility(ISecurityManager)
                found, principals = boolean_iter(sm.find_principals(owner))
                if found:
                    owner = next(principals).id
            query &= Eq(catalog['role:owner'], owner)
        if 'age_limit' in params:
            now = tztime(datetime.utcnow())
            age_limit = params['age_limit']
            params &= Gt(catalog['content_publication_date'],
                         now - timedelta(days=age_limit))
        return query
