#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from cornice import Service
from cornice.validators import colander_validator
from pyramid.httpexceptions import HTTPBadRequest, HTTPForbidden, HTTPNotFound, HTTPOk, HTTPServiceUnavailable, \
    HTTPUnauthorized
from pyramid.security import Authenticated

from pyams_content.features.json import to_json
from pyams_content.interfaces import USE_PUBLIC_API_PERMISSION
from pyams_content.shared.common.api.schema import ContentGetterRequest, ContentGetterResponse, ContentSearchRequest, \
    ContentSearchResponse, DataTypesResponse, DisplayName, SharedContentInfo
from pyams_content.shared.common.api.interfaces import REST_CONTENT_PUBLIC_GETTER_ROUTE, \
    REST_CONTENT_PUBLIC_GETTER_ROUTE_DEFAULT, REST_CONTENT_PUBLIC_GETTER_ROUTE_SETTING, \
    REST_CONTENT_PUBLIC_SEARCH_ROUTE, REST_DATATYPES_PUBLIC_GETTER_ROUTE
from pyams_content.shared.common.interfaces import ISharedTool
from pyams_content.shared.common.interfaces.types import ITypedDataManager
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_sequence.reference import get_reference_target, get_visible_version
from pyams_utils.registry import query_utility
from pyams_utils.rest import http_error, http_response, rest_responses
from pyams_workflow.interfaces import IWorkflow, IWorkflowState


__docformat__ = 'restructuredtext'


content_datatypes_service = Service(name=REST_DATATYPES_PUBLIC_GETTER_ROUTE,
                                    pyramid_route=REST_DATATYPES_PUBLIC_GETTER_ROUTE,
                                    description="Shared content data types getter service")


@content_datatypes_service.options(validators=(check_cors_origin, set_cors_headers))
def content_datatypes_service_options(request):
    return ''



content_datatypes_responses = rest_responses.copy()
content_datatypes_responses[HTTPOk.code] = DataTypesResponse(description="Content datatypes results")


@content_datatypes_service.get(validators=(check_cors_origin, colander_validator, set_cors_headers),
                               response_schemas=content_datatypes_responses)
def content_datatypes_service_get(request):
    """Shared content datatypes getter"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    content_type = request.matchdict['content_type']
    if not content_type:
        return http_error(request, HTTPBadRequest)
    manager = query_utility(ISharedTool, name=content_type)
    if manager is None:
        return http_error(request, HTTPServiceUnavailable)
    data_types = ITypedDataManager(manager, None)
    if data_types is None:
        return http_error(request, HTTPServiceUnavailable)
    return http_response(request, HTTPOk, {
        'results': [
            to_json(data_type, request)
            for data_type in data_types.values()
        ]
    })


#
# Generic content search service
#

content_search_service = Service(name=REST_CONTENT_PUBLIC_SEARCH_ROUTE,
                                 pyramid_route=REST_CONTENT_PUBLIC_SEARCH_ROUTE,
                                 description="Shared content search service")


@content_search_service.options(validators=(check_cors_origin, set_cors_headers))
def content_search_service_options(request):
    return ''


content_search_responses = rest_responses.copy()
content_search_responses[HTTPOk.code] = ContentSearchResponse(description="Content search results")


@content_search_service.get(schema=ContentSearchRequest(),
                            validators=(check_cors_origin, colander_validator, set_cors_headers),
                            response_schemas=content_search_responses)
def content_search_service_get(request):
    """Content search service getter"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    content_type = request.matchdict['content_type']
    if not content_type:
        return http_error(request, HTTPBadRequest)
    manager = query_utility(ISharedTool, name=content_type)
    if manager is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(USE_PUBLIC_API_PERMISSION, context=manager):
        return http_error(request, HTTPForbidden)
    workflow = IWorkflow(manager)
    params = request.validated.get('querystring', {})
    params.update({
        'state': workflow.visible_states
    })
    return http_response(request, HTTPOk, {
        'results': [
            content.to_json(params)
            for content in manager.find(params, request)
        ]
    })


#
# Generic content getter service
#

content_getter_service = Service(name=REST_CONTENT_PUBLIC_GETTER_ROUTE,
                                 pyramid_route=REST_CONTENT_PUBLIC_GETTER_ROUTE,
                                 description="Shared content getter service")

content_getter_responses = rest_responses.copy()
content_getter_responses[HTTPOk.code] = ContentGetterResponse(description="Content getter result")


@content_getter_service.options(validators=(check_cors_origin, set_cors_headers))
def content_getter_service_options(request):
    return ''


@content_getter_service.get(schema=ContentGetterRequest(),
                            validators=(check_cors_origin, colander_validator, set_cors_headers),
                            response_schemas=content_getter_responses)
def content_getter_service_get(request):
    """Content getter service"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    content_type = request.matchdict['content_type']
    if not content_type:
        return http_error(request, HTTPBadRequest)
    manager = query_utility(ISharedTool, name=content_type)
    if manager is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(USE_PUBLIC_API_PERMISSION, context=manager):
        return http_error(request, HTTPForbidden)
    oid = request.matchdict['oid']
    if not oid:
        return http_error(request, HTTPBadRequest)
    content = get_reference_target(oid, request=request, getter=get_visible_version)
    if content is None:
        return http_error(request, HTTPNotFound)
    wf_state = IWorkflowState(content, None)
    if wf_state is not None:
        workflow = IWorkflow(content)
        if wf_state.state not in workflow.published_states:
            return http_error(request, HTTPNotFound)
    params = request.validated.get('querystring')
    return http_response(request, HTTPOk, {
        'content': content.to_json(params)
    })
