#
# Copyright (c) 2008-2016 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from datetime import datetime

from pyramid.events import subscriber
from pyramid.path import DottedNameResolver
from z3c.form import field
from z3c.form.interfaces import IDataExtractedEvent
from zope.interface import Invalid

from pyams_content.interfaces import MANAGE_SITE_PERMISSION, MANAGE_SITE_ROOT_PERMISSION, \
    MANAGE_TOOL_PERMISSION
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.blog.interfaces import IBlogManager, IBlogManagerFactory
from pyams_content.shared.blog.manager import BlogManager
from pyams_content.shared.common import IBaseSharedTool, IDashboardTypeColumnValue
from pyams_content.shared.common.interfaces.zmi import IDashboardTable
from pyams_content.shared.common.zmi.manager import SharedToolPropertiesEditForm
from pyams_content.zmi.interfaces import ISiteTreeTable, IUserAddingsMenuLabel
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.interfaces.viewlet import IMenuHeader, ITableItemColumnActionsMenu, \
    IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import DefaultElementEditorAdapter
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.registry import query_utility
from pyams_utils.timezone import tztime
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflowPublicationInfo
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu, ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@adapter_config(context=(IDashboardTable, IPyAMSLayer, IBlogManager),
                provides=IDashboardTypeColumnValue)
def blog_manager_dashboard_type(context, request):
    """Blog manager dashboard type column adapter"""
    return request.localizer.translate(context.content_name)


@adapter_config(context=(IBlogManager, ISiteManagementMenu), provides=IMenuHeader)
class BlogManagerSiteManagementMenuHeader(ContextRequestAdapter):
    """Blog manager site management menu header adapter"""

    header = _("Blog management")


#
# Blog topic add views
#

@adapter_config(context=(IBlogManager, IAdminLayer), provides=IUserAddingsMenuLabel)
class BlogManagerUserAddingsMenuLabelAdapter(ContextRequestAdapter):
    """Blog manager user addings menu label adapter"""

    @property
    def label(self):
        return '{content} ({blog})'.format(
            content=self.request.localizer.translate(
                self.context.shared_content_factory.content_class.content_name),
            blog=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-blog-manager.menu', context=ISiteRoot, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class BlogManagerAddMenu(ToolbarMenuItem):
    """Blog manager add menu"""

    label = _("Add blog manager")
    label_css_class = 'fa fa-fw fa-tags'
    url = 'add-blog-manager.html'
    modal_target = True


@pagelet_config(name='add-blog-manager.html', context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-blog-manager.json', context=ISiteRoot, layer=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, base=AJAXAddForm)
class BlogManagerAddForm(AdminDialogAddForm):
    """Blog manager add form"""

    title = _("Blog manager")
    legend = _("Add blog manager")
    icon_css_class = 'fa fa-fw fa-tags'

    fields = field.Fields(IBlogManager).select('title', 'short_name')
    edit_permission = None

    def create(self, data):
        factory = self.request.registry.settings.get('pyams_content.config.blog_factory')
        if factory:
            factory = DottedNameResolver().resolve(factory)
        else:
            factory = IBlogManagerFactory(self.context, BlogManager)
        return factory()

    def add(self, object):
        short_name = II18n(object).query_attribute('short_name', request=self.request)
        name = translate_string(short_name, force_lower=True, spaces='-')
        self.context[name] = object

    def nextURL(self):
        return absolute_url(self.context, self.request, 'site-tree.html')


@subscriber(IDataExtractedEvent, form_selector=BlogManagerAddForm)
def handle_new_blog_manager_data_extraction(event):
    """Handle new blog manager data extraction"""
    container = event.form.context
    negotiator = query_utility(INegotiator)
    short_name = event.data['short_name'].get(negotiator.server_language)
    if not short_name:
        event.form.widgets.errors += (Invalid(_("You must provide a short name for default "
                                                "server language!")),)
        return
    name = translate_string(short_name, force_lower=True, spaces='-')
    if name in container:
        event.form.widgets.errors += (Invalid(_("Specified blog manager name is already "
                                                "used!")),)
        return
    blog = query_utility(IBlogManager, name=short_name)
    if blog is not None:
        event.form.widgets.errors += (Invalid(_("A blog manager is already registered with "
                                                "this name!!")),)


@adapter_config(context=(IBlogManager, IAdminLayer, ISiteTreeTable),
                provides=ITableElementEditor)
class BlogManagerTableElementEditor(DefaultElementEditorAdapter):
    """Blog manager table element editor"""

    view_name = 'admin'
    modal_target = False


@pagelet_config(name='properties.html', context=IBlogManager, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json', context=IBlogManager, layer=IPyAMSLayer)
class BlogManagerPropertiesEditForm(SharedToolPropertiesEditForm):
    """Site folder properties edit form"""

    legend = _("Blog properties")

    fields = field.Fields(IBlogManager).select('title', 'short_name', 'description',
                                               'notepad') + \
        field.Fields(IBaseSharedTool).select('shared_content_workflow')


#
# Blog manager publication views
#

@viewlet_config(name='workflow-publication.menu', context=IBlogManager, layer=IAdminLayer,
                view=ISiteTreeTable, manager=ITableItemColumnActionsMenu,
                permission=MANAGE_SITE_PERMISSION, weight=510)
class BlogManagerTableItemWorkflowPublicationMenu(ToolbarMenuItem):
    """Blog manager tree item workflow publication menu"""

    label = _("Publication dates...")
    label_css_class = 'fa fa-fw fa-eye'
    url = 'workflow-publication.html'
    modal_target = True
    stop_propagation = True


@viewlet_config(name='workflow-publication.menu', context=IBlogManager, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_SITE_PERMISSION, weight=510)
class BlogManagerWorkflowPublicationMenu(MenuItem):
    """Blog manager workflow publication menu"""

    label = _("Publication dates...")
    icon_class = 'fa-eye'
    url = 'workflow-publication.html'
    modal_target = True


@pagelet_config(name='workflow-publication.html', context=IBlogManager, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='workflow-publication.json', context=IBlogManager, layer=IPyAMSLayer)
class BlogManagerWorkflowPublicationEditForm(AdminDialogEditForm):
    """Blog manager workflow publication edit form"""

    prefix = 'blog_publication.'

    legend = _("Update publication dates")

    fields = field.Fields(IWorkflowPublicationInfo).select('publication_effective_date',
                                                           'publication_expiration_date')
    edit_permission = MANAGE_SITE_PERMISSION

    def updateWidgets(self, prefix=None):
        super(BlogManagerWorkflowPublicationEditForm, self).updateWidgets(prefix)
        if 'publication_effective_date' in self.widgets:
            pub_info = IWorkflowPublicationInfo(self.context)
            if pub_info.publication_effective_date is None:
                self.widgets['publication_effective_date'].value = \
                    tztime(datetime.utcnow()).strftime('%d/%m/%y %H:%M')
