#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from persistent import Persistent
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_content.reference.pictograms.interfaces import IPictogramTable
from pyams_content.shared.alert.interfaces import IAlertType, IAlertTypesManager, IAlertsManager, \
    ALERT_TYPES_MANAGER_ANNOTATION_KEY, ALERT_TYPES_VOCABULARY
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config, get_annotation_adapter, ContextAdapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_utils.vocabulary import vocabulary_config


@factory_config(IAlertType)
class AlertType(Persistent, Contained):
    """Alert type"""

    label = FieldProperty(IAlertType['label'])
    backoffice_label = FieldProperty(IAlertType['backoffice_label'])
    pictogram = FieldProperty(IAlertType['pictogram'])
    color = FieldProperty(IAlertType['color'])

    def get_pictogram(self):
        table = query_utility(IPictogramTable)
        return table.get(self.pictogram) if table is not None else None


@adapter_config(required=(IAlertType, IPyAMSLayer),
                provides=IJSONConverter)
class AlertTypeJSONConverter(JSONBaseConverter):
    """Alert type JSON converter"""

    def to_json(self, params=None):
        context = self.context
        request = self.request
        result = {
            'name': context.__name__,
            'label': II18n(context).query_attribute('label', request=request),
            'color': '#{}'.format(context.color)
        }
        pictogram = context.get_pictogram()
        if pictogram is not None:
            image = II18n(pictogram).query_attribute('image', request=request)
            if image:
                result['pictogram'] = {
                    'content_type': image.content_type,
                    'src': absolute_url(image, request)
                }
        return result


@factory_config(IAlertTypesManager)
class AlertTypesManager(OrderedContainer):
    """Alert types manager"""


@adapter_config(required=IAlertsManager,
                provides=IAlertTypesManager)
def alerts_types_manager(context):
    """Alerts types manager adapter"""
    return get_annotation_adapter(context, ALERT_TYPES_MANAGER_ANNOTATION_KEY, IAlertTypesManager,
                                  name='++types++')


@adapter_config(name='types',
                required=IAlertsManager,
                provides=ITraversable)
class AlertsManagerTypesNamespace(ContextAdapter):
    """Alerts manager types ++gravity++ namespace"""

    def traverse(self, name, furtherpath=None):
        return IAlertTypesManager(self.context)


@adapter_config(name='types',
                required=IAlertsManager,
                provides=ISublocations)
class AlertsManagerTypesSublocations(ContextAdapter):
    """Alerts manager types sub-locations adapter"""

    def sublocations(self):
        return IAlertTypesManager(self.context).values()


@vocabulary_config(name=ALERT_TYPES_VOCABULARY)
class AlertTypesVocabulary(SimpleVocabulary):
    """Alert types vocabulary"""

    def __init__(self, context):
        terms = []
        parent = get_parent(context, IAlertsManager)
        if parent is not None:
            request = check_request()
            manager = IAlertTypesManager(parent)
            terms = [
                SimpleTerm(alert_type.__name__,
                           title=II18n(alert_type).query_attribute('backoffice_label', request=request) or
                                 II18n(alert_type).query_attribute('label', request=request))
                for alert_type in manager.values()
            ]
        super().__init__(terms)
