#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from zope.interface import implementer, provider
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.links import InternalReferenceMixin
from pyams_content.component.theme.interfaces import ITagsTarget, IThemesTarget
from pyams_content.shared.alert.interfaces import ALERT_CONTENT_NAME, ALERT_CONTENT_TYPE, \
    IAlert, IWfAlert, IWfAlertFactory, IAlertsManager, IAlertTypesManager
from pyams_content.shared.common import SharedContent, WfSharedContent, register_content_type
from pyams_content.shared.common.interfaces import IWfSharedContentCreator, IWfSharedContentFactory, \
    IWfSharedContentUpdater
from pyams_i18n.interfaces import INegotiator
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import get_utility
from pyams_utils.request import check_request


__docformat__ = 'restructuredtext'

from pyams_utils.traversing import get_parent


@implementer(IWfAlert, ITagsTarget, IThemesTarget)
class WfAlert(WfSharedContent, InternalReferenceMixin):
    """Alert persistent class"""

    content_type = ALERT_CONTENT_TYPE
    content_name = ALERT_CONTENT_NAME

    handle_content_url = False
    handle_header = False
    handle_description = False

    alert_type = FieldProperty(IWfAlert['alert_type'])
    body = FieldProperty(IWfAlert['body'])
    _reference = FieldProperty(IWfAlert['reference'])
    external_url = FieldProperty(IWfAlert['external_url'])
    references = FieldProperty(IWfAlert['references'])
    maximum_interval = FieldProperty(IWfAlert['maximum_interval'])

    @property
    def reference(self):
        return self._reference

    @reference.setter
    def reference(self, value):
        self._reference = value
        del self.target

    def get_alert_type(self):
        manager = get_parent(self, IAlertsManager)
        types = IAlertTypesManager(manager, None)
        if types is not None:
            return types.get(self.alert_type)

    def get_targets(self, state=None):
        request = check_request()
        return [
            get_reference_target(reference, state, request)
            for reference in (self.references or ())
        ]


register_content_type(WfAlert, shared_content=False)


@provider(IWfAlertFactory)
@implementer(IAlert)
class Alert(SharedContent):
    """Workflow manager alert persistent class"""


@adapter_config(required=IWfAlertFactory,
                provides=IWfSharedContentFactory)
def alert_content_factory(context):
    return WfAlert


@adapter_config(name='alert',
                required=IWfAlert,
                provides=IWfSharedContentCreator)
@adapter_config(name='alert',
                required=IWfAlert,
                provides=IWfSharedContentUpdater)
class AlertContentCreator(ContextAdapter):
    """Alert content creator"""

    weight = 500

    def update_content(self, params, request=None):
        """Create alert from REST request params"""
        alert = self.context
        if 'alert_type' in params:
            alert.alert_type = params.get('alert_type')
        if 'body' in params:
            negotiator = get_utility(INegotiator)
            lang = negotiator.server_language
            body = params['body']
            alert.body = body if isinstance(body, dict) else {lang: body}
        if 'href' in params:
            alert.external_url = params['href']
        if 'maximum_interval' in params:
            alert.maximum_interval = params['maximum_interval']
