#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.exceptions import NotFound
from pyramid.view import view_config
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import implementer
from zope.intid.interfaces import IIntIds

from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.common.interfaces import IBaseSharedTool, ISharedSite
from pyams_content.shared.common.interfaces.zmi import IDashboardTable
from pyams_content.zmi import pyams_content
from pyams_content.zmi.interfaces import ISiteTreeMenu, ISiteTreeTable
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.container import ContainerView, delete_container_element
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableWithActions
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import ActionColumn, BaseTable, I18nColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_utility
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflowPublicationInfo
from pyams_zmi.interfaces.menu import ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Sites and blogs view
#

@viewlet_config(name='site-tree.menu', context=ISiteRoot, layer=IAdminLayer, manager=ISiteManagementMenu,
                permission=VIEW_SYSTEM_PERMISSION, weight=10)
@viewletmanager_config(name='site-tree.menu', layer=IAdminLayer, context=ISiteRoot, provides=ISiteTreeMenu)
@implementer(ISiteTreeMenu)
class SiteTreeMenu(MenuItem):
    """Site tree menu"""

    label = _("Site tree")
    icon_class = 'fa-sitemap'
    url = '#site-tree.html'


@implementer(IDashboardTable, ITableWithActions, ISiteTreeTable)
class SiteTreeTable(BaseTable):
    """Site tree table"""

    id = 'site_tree_table'
    title = _("Blogs and shared sites")

    sortOn = None

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        intids = get_utility(IIntIds)
        attributes.setdefault('table', {}).update({
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(self.context, self.request)
        })
        attributes.setdefault('tr', {}).update({
            'id': lambda x, col: '{0}::{1}'.format(self.id, intids.queryId(x)),
            'data-ams-location': lambda x, col: absolute_url(x.__parent__, self.request),
            'data-ams-tree-node-id': lambda x, col: intids.queryId(x),
            'data-ams-tree-node-parent-id': lambda x, col: intids.queryId(x.__parent__)
        })
        return attributes


@adapter_config(name='visible', context=(ISiteRoot, IPyAMSLayer, SiteTreeTable), provides=IColumn)
class SiteTreeVisibleColumn(ActionColumn):
    """Site tree visible column"""

    cssClasses = {'th': 'action',
                  'td': 'action'}

    icon_class = 'fa fa-fw fa-eye'
    icon_hint = _("Visible site?")

    weight = 5

    def renderCell(self, item):
        return self.get_icon(item)

    def get_icon(self, item):
        info = IWorkflowPublicationInfo(item, None)
        if info is None:
            return ''
        else:
            if info.is_published():
                icon_class = 'fa-eye opacity-75'
            else:
                icon_class = 'fa-eye-slash text-danger opaque'
            return '<i class="fa fa-fw {icon_class} hint align-base" title="{title}" ' \
                   'data-ams-hint-gravity="e"></i>'.format(
                    icon_class=icon_class,
                    title=self.request.localizer.translate(self.icon_hint))


@adapter_config(name='oid', context=(ISiteRoot, IPyAMSLayer, SiteTreeTable), provides=IColumn)
class SiteTreeOidColumn(I18nColumn, GetAttrColumn):
    """Site tree OID column"""

    _header = _("OID")
    weight = 70

    def getValue(self, obj):
        sequence = ISequentialIdInfo(obj, None)
        if sequence is None:
            return '--'
        else:
            try:
                return sequence.get_base_oid()
            except TypeError:
                return '--'


@adapter_config(name='trash', context=(ISiteRoot, IAdminLayer, SiteTreeTable), provides=IColumn)
class SiteTreeTrashColumn(TrashColumn):
    """Site tree trash column"""

    icon_hint = _("Delete shared site")
    permission = MANAGE_SITE_ROOT_PERMISSION

    def has_permission(self, item):
        return super(SiteTreeTrashColumn, self).has_permission(item) and item.is_deletable()


@view_config(name='delete-element.json', context=ISiteRoot, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, renderer='json', xhr=True)
def delete_shared_site(request):
    """Delete shared site from site manager"""
    translate = request.localizer.translate
    # Get object name to be removed
    name = request.params.get('object_name')
    if not name:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("No provided object_name argument!"))
            }
        }
    # Check context
    if name not in request.context:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("Given element name doesn't exist!"))
            }
        }
    context = request.context[name]
    if not context.is_deletable():
        raise NotFound()
    return delete_container_element(request, ignore_permission=True)


@adapter_config(context=(ISiteRoot, IAdminLayer, SiteTreeTable), provides=IValues)
class SiteTreValuesAdapter(ContextRequestViewAdapter):
    """Site tree values adapter"""

    @property
    def values(self):
        return [value for value in self.context.values()
                if IBaseSharedTool.providedBy(value) or ISharedSite.providedBy(value)]


@pagelet_config(name='site-tree.html', context=ISiteRoot, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteTreeView(AdminView, ContainerView):
    """Site tree view"""

    table_class = SiteTreeTable

    def __init__(self, context, request):
        super(SiteTreeView, self).__init__(context, request)


@adapter_config(context=(ISiteRoot, IAdminLayer, SiteTreeView), provides=IPageHeader)
class SiteViewHeaderAdapter(DefaultPageHeaderAdapter):
    """Site tree view header adapter"""

    icon_class = 'fa fa-fw fa-sitemap'
