#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from itertools import chain

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Eq
from pyramid.decorator import reify
from z3c.form.button import Button, Buttons
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer
from zope.intid.interfaces import IIntIds

from pyams_catalog.query import CatalogResultSet
from pyams_catalog.utils import reindex_object
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.root import ISiteRoot
from pyams_content.shared.common import IWfSharedContentRoles
from pyams_form.form import ajax_config
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.interfaces import IRoleProtectedObject
from pyams_security.principal import MissingPrincipal
from pyams_security.utility import get_principal
from pyams_security.zmi.interfaces import IObjectSecurityMenu
from pyams_sequence.interfaces import ISequentialIdInfo, ISequentialIdTarget
from pyams_skin.container import ContainerView
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import BaseTable, I18nColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import MANAGE_SYSTEM_PERMISSION
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowState
from pyams_zmi.form import InnerAdminEditForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='contributors-checker.menu',
                context=ISiteRoot, layer=IAdminLayer,
                manager=IObjectSecurityMenu, weight=20,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class ContributorsCheckerMenu(MenuItem):
    """Contributors checker menu"""

    label = _("Roles checker")
    icon_class = 'fa-user-times'
    url = '#contributors-checker.html'


class ContributorsCheckerTable(BaseTable):
    """Contributors checker table"""

    id = 'contributors'
    title = _("Missing principals with roles")

    sortOn = None

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('tr', {}).update({'data-ams-target': '_blank'})
        return attributes


ROLE_INDEX_NAMES = (
    'role:contributor',
    'role:manager',
    'role:pilot',
    'role:webmaster'
)

ROLE_INDEX_ROLES = {
    'role:contributor': 'pyams.Contributor',
    'role:manager': 'pyams.Manager',
    'role:pilot': 'pyams.Pilot',
    'role:webmaster': 'pyams.Webmaster'
}

ROLE_INDEX_LABEL = {
    'role:contributor': _("Contributor"),
    'role:manager': _("Manager"),
    'role:pilot': _("Pilot"),
    'role:webmaster': _("Webmaster")
}


@adapter_config(context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IValues)
class ContributorsCheckerValues(ContextRequestViewAdapter):
    """Contributors checker values adapter"""

    @property
    def values(self):
        catalog = get_utility(ICatalog)
        intids = get_utility(IIntIds)
        for principal_id in sorted(set(chain(*(index.unique_values()
                                               for index in (catalog[index_name]
                                                             for index_name in ROLE_INDEX_NAMES))))):
            principal = get_principal(self.request, principal_id=principal_id)
            if not isinstance(principal, MissingPrincipal):
                continue
            for index_name in ROLE_INDEX_NAMES:
                query = Eq(catalog[index_name], principal_id)
                for context in CatalogResultSet(CatalogQuery(catalog).query(query)):
                    workflow = IWorkflow(context, None)
                    state = IWorkflowState(context, None)
                    if (workflow is not None) and \
                            (state is not None) and \
                            (state.state in workflow.archived_states):
                        continue
                    is_owner = False
                    nb_docs = 0
                    if index_name == 'role:contributor':
                        roles = IWfSharedContentRoles(context, None)
                        if (roles is not None) and (principal_id in roles.owner):
                            is_owner = True
                            nb_docs = 1
                        else:
                            params = And(Eq(catalog['role:owner'], principal_id),
                                         Eq(catalog['parents'], intids.register(context)))
                            nb_docs, items = CatalogQuery(catalog).query(params)
                    yield principal, index_name, context, is_owner, nb_docs


@adapter_config(context=(Interface, IAdminLayer, ContributorsCheckerTable),
                provides=ITableElementEditor)
class ContributorsCheckerTableElementEditor(ContextRequestViewAdapter):
    """Contributors checker table element editor"""

    @property
    def url(self):
        return absolute_url(self.context[2], self.request, 'admin#protected-object-roles.html')

    modal_target = False


@adapter_config(name='name',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerPrincipalIdColumn(I18nColumn, GetAttrColumn):
    """Contributors checker principal ID column"""

    _header = _("Principal ID")
    attrName = 'id'

    weight = 10

    def getValue(self, obj):
        return super().getValue(obj[0])


@adapter_config(name='role',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerRoleColumn(I18nColumn, GetAttrColumn):
    """Contributors checker role column"""

    _header = _("Role name")

    weight = 20

    def getValue(self, obj):
        return self.request.localizer.translate(ROLE_INDEX_LABEL[obj[1]])


@adapter_config(name='context',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerContextColumn(I18nColumn, GetAttrColumn):
    """Contributors checker context column"""

    _header = _("Context")

    weight = 30

    def getValue(self, obj):
        context = obj[2]
        return II18n(context).query_attribute('title', request=self.request)


@adapter_config(name='oid',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerOidColumn(I18nColumn, GetAttrColumn):
    """Contributors checker OID column"""

    _header = _("Unique ID")

    weight = 35

    def getValue(self, obj):
        context = obj[2]
        sequence = ISequentialIdInfo(context, None)
        if sequence is not None:
            target = get_parent(context, ISequentialIdTarget)
            if target is not None:
                return sequence.get_base_oid()
        return '--'


@adapter_config(name='version',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerVersionColumn(I18nColumn, GetAttrColumn):
    """Contributors checker version column"""

    _header = _("Version")

    weight = 40

    def getValue(self, obj):
        context = obj[2]
        state = IWorkflowState(context, None)
        if state is None:
            return '--'
        return state.version_id


@adapter_config(name='state',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerStateColumn(I18nColumn, GetAttrColumn):
    """Contributors checker state column"""

    _header = _("Status")

    weight = 50

    def getValue(self, obj):
        context = obj[2]
        state = IWorkflowState(context, None)
        if state is None:
            return '--'
        return self.request.localizer.translate(IWorkflow(context).get_state_label(state.state))


@adapter_config(name='is_owner',
                context=(ISiteRoot, IAdminLayer, ContributorsCheckerTable),
                provides=IColumn)
class ContributorsCheckerNbDocsColumn(I18nColumn, GetAttrColumn):
    """Contributors checker owned documents count column"""

    _header = _("Owned")
    _hint = _("A check indicates that the matching principal is owner of one or more "
              "contents in the given context")

    weight = 60

    def renderHeadCell(self):
        result = super().renderHeadCell()
        return '{} <i class="fa fa-question-circle hint" original-title="{}">'.format(
            result,
            self.request.localizer.translate(self._hint)
        )

    def getValue(self, obj):
        if obj[3]:
            return '<i class="fa fa-fw fa-check"></i>'
        return obj[4] or '--'


class IContributorsCheckerSubmitActions(Interface):
    """Contributors checker submit actions interface"""

    submit = Button(name='submit', title=_("Remove roles from missing principals"))


@pagelet_config(name='contributors-cleanup.html',
                context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
@ajax_config(name='contributors-cleanup.json',
             context=ISiteRoot, layer=IPyAMSLayer)
class ContributorsCheckerSubmitForm(InnerAdminEditForm):
    """Contributors checker submit form"""

    title = None
    legend = None

    buttons = Buttons(IContributorsCheckerSubmitActions)

    edit_permission = MANAGE_SYSTEM_PERMISSION

    def render(self):
        if not self.request.has_permission(self.edit_permission, context=self.context):
            return ''
        return '''<div class="ams-widget">
            <form class="ams-form"
                  method="post"
                  action="contributors.html"
                  data-ams-form-handler="{}"
                  data-async>
                {}
            </form>
        </div>'''.format(absolute_url(self.context, self.request, 'contributors-cleanup.json'),
                         super().render())

    def update_content(self, content, data):
        updates = set()
        catalog = get_utility(ICatalog)
        for principal_id in sorted(set(chain(*(index.unique_values()
                                               for index in (
                                                   catalog[index_name]
                                                   for index_name in ROLE_INDEX_NAMES
                                               ))))):
            principal = get_principal(self.request, principal_id=principal_id)
            if not isinstance(principal, MissingPrincipal):
                continue
            for index_name in ROLE_INDEX_NAMES:
                query = Eq(catalog[index_name], principal_id)
                for context in CatalogResultSet(CatalogQuery(catalog).query(query)):
                    workflow = IWorkflow(context, None)
                    state = IWorkflowState(context, None)
                    if (workflow is not None) and \
                            (state is not None) and \
                            (state.state in workflow.archived_states):
                        continue
                    roles = IRoleProtectedObject(context, None)
                    if roles is None:
                        continue
                    roles.revoke_role(ROLE_INDEX_ROLES[index_name], principal_id)
                    updates.add(context)
        for updated in updates:
            reindex_object(updated)

    def get_ajax_output(self, changes):
        return {
            'status': 'reload',
            'message': self.successMessage
        }


@pagelet_config(name='contributors-checker.html',
                context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@implementer(IInnerPage)
class ContributorsCheckerView(AdminView, ContainerView):
    """Contributors checker view"""

    table_class = ContributorsCheckerTable

    def __init__(self, context, request):
        super().__init__(context, request)
        self.footer = ContributorsCheckerSubmitForm(self.context, self.request, self)

    def update(self):
        super().update()
        self.footer.update()

    def render(self):
        return super().render() + self.footer.render()


@adapter_config(context=(ISiteRoot, IAdminLayer, ContributorsCheckerView),
                provides=IPageHeader)
class ContributorsCheckerViewHeaderAdapter(DefaultPageHeaderAdapter):
    """Contributors checker view header adapter"""

    icon_class = 'fa fa-fw fa-user-times'
