#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Any, Eq, Or
from z3c.table.interfaces import IValues
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, implementer
from zope.intid.interfaces import IIntIds

from pyams_catalog.query import CatalogResultSet
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION, PUBLISH_CONTENT_PERMISSION
from pyams_content.profile.interfaces import IAdminProfile
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.common import CONTENT_TYPES
from pyams_content.shared.common.interfaces import IBaseSharedTool, IManagerRestrictions
from pyams_content.shared.common.interfaces.zmi import ISiteRootDashboardTable
from pyams_content.shared.common.zmi.dashboard import BaseDashboardTable as BaseDashboardTableBase
from pyams_content.zmi.interfaces import IAllContentsMenu, IDashboardMenu, IMyDashboardMenu
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.reference import get_last_version
from pyams_skin.container import ContainerView
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.configuration import IBackOfficeConfiguration, IConfiguration
from pyams_skin.interfaces.container import ITableElementEditor, ITableElementName
from pyams_skin.interfaces.viewlet import IBreadcrumbItem
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import DefaultElementEditorAdapter
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.list import unique_iter
from pyams_utils.registry import get_all_utilities_registered_for, get_utility
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowState, IWorkflowVersions
from pyams_zmi.interfaces.menu import IContentManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


@implementer(ISiteRootDashboardTable)
class BaseDashboardTable(BaseDashboardTableBase):
    """Base dashboard table"""


#
# Site root breadcrumbs
#

@adapter_config(context=(ISiteRoot, IAdminLayer), provides=IBreadcrumbItem)
class SiteRootBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """Site root breadcrumb adapter"""

    label = _("Home")


@adapter_config(context=(ISiteRoot, IPyAMSLayer), provides=ITableElementName)
class SiteRootTableElementNameAdapter(ContextRequestAdapter):
    """Site root table element name adapter"""

    @property
    def name(self):
        configuration = IConfiguration(self.context)
        return II18n(configuration).query_attribute('short_title', request=self.request)


@adapter_config(context=(ISiteRoot, IAdminLayer, Interface), provides=ITableElementEditor)
class SiteRootTableElementEditorAdapter(DefaultElementEditorAdapter):
    """Site root table element editor adapter"""

    view_name = 'admin#site-tree.html'
    modal_target = False


#
# Main dashboard menu
#

@viewlet_config(name='dashboard.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IContentManagementMenu, permission=VIEW_SYSTEM_PERMISSION, weight=1)
@viewletmanager_config(name='dashboard.menu', layer=IAdminLayer, provides=IDashboardMenu)
@implementer(IDashboardMenu)
class SiteRootDashboardMenu(MenuItem):
    """Site root dashboard menu"""

    label = _("Dashboard")
    icon_class = 'fa-line-chart'
    url = '#dashboard.html'


@pagelet_config(name='dashboard.html', context=ISiteRoot, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@template_config(template='templates/dashboard.pt', layer=IAdminLayer)
@implementer(IInnerPage)
class SiteRootDashboardView(AdminView):
    """Site root dashboard view"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    def __init__(self, context, request):
        super(SiteRootDashboardView, self).__init__(context, request)
        self.tables = []
        self.tables.append(SiteRootDashboardManagerWaitingTable(self.context, self.request))
        self.tables.append(SiteRootDashboardOwnerWaitingTable(self.context, self.request))
        self.tables.append(SiteRootDashboardOwnerModifiedTable(self.context, self.request))
        for table in self.tables:
            table.hide_toolbar = True

    def update(self):
        super(SiteRootDashboardView, self).update()
        [table.update() for table in self.tables]


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootDashboardView), provides=IPageHeader)
class SiteRootDashboardHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root properties header adapter"""

    icon_class = 'fa fa-fw fa-line-chart'

    @property
    def title(self):
        config = IBackOfficeConfiguration(self.context)
        return II18n(config).query_attribute('title', request=self.request)


#
# Contents waiting for manager action
#

@implementer(ISiteRootDashboardTable)
class SiteRootDashboardManagerWaitingTable(BaseDashboardTable):
    """Site root dashboard manager waiting contents table"""

    _single_title = _("MANAGER - {0} content waiting for your action")
    _plural_title = _("MANAGER - {0} contents waiting for your action")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootDashboardManagerWaitingTable), provides=IValues)
class SiteRootDashboardManagerWaitingValues(ContextRequestViewAdapter):
    """Site root dashboard manager waiting contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = Eq(catalog['parents'], intids.register(tool)) & \
                Any(catalog['content_type'], CONTENT_TYPES.keys()) & \
                Any(catalog['workflow_state'], workflow.waiting_states)
            params = params | query if params else query
        yield from filter(self.check_access,
                          unique_iter(map(lambda x: sorted(IWorkflowVersions(x).get_versions(IWorkflowState(x).state),
                                                           key=lambda y: IZopeDublinCore(y).modified, reverse=True)[0],
                                      CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                                   sort_index='modified_date')))))

    def check_access(self, content):
        if self.request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=content):
            return True
        if self.request.principal.id in content.managers:
            return True
        restrictions = IManagerRestrictions(content).get_restrictions(self.request.principal)
        if restrictions is not None:
            return restrictions.check_access(content, PUBLISH_CONTENT_PERMISSION, self.request)
        else:
            return False


#
# Last owned contents waiting for action
#

@implementer(ISiteRootDashboardTable)
class SiteRootDashboardOwnerWaitingTable(BaseDashboardTable):
    """Site root dashboard waiting owned contents table"""

    _single_title = _("CONTRIBUTOR - {0} content waiting for action")
    _plural_title = _("CONTRIBUTOR - {0} contents waiting for action")

    dt_sort_order = 'asc'


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootDashboardOwnerWaitingTable), provides=IValues)
class SiteRootDashboardOwnerWaitingValues(ContextRequestViewAdapter):
    """Site root dashboard waiting owned contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = Eq(catalog['parents'], intids.register(tool)) & \
                Any(catalog['content_type'], CONTENT_TYPES.keys()) & \
                Any(catalog['workflow_state'], workflow.waiting_states) & \
                Eq(catalog['workflow_principal'], self.request.principal.id)
            params = params | query if params else query
        yield from unique_iter(map(lambda x: sorted(IWorkflowVersions(x).get_versions(IWorkflowState(x).state),
                                                    key=lambda y: IZopeDublinCore(y).modified, reverse=True)[0],
                                   CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                                sort_index='modified_date'))))


#
# Last owned modified contents
#

@implementer(ISiteRootDashboardTable)
class SiteRootDashboardOwnerModifiedTable(BaseDashboardTable):
    """Site root dashboard modified contents table"""

    _single_title = _("CONTRIBUTOR - {0} modified content")

    @property
    def _plural_title(self):
        if len(self.values) <= 50:
            return _("CONTRIBUTOR - {0} modified contents")
        else:
            return _("CONTRIBUTOR - Last {0} modified contents")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootDashboardOwnerModifiedTable), provides=IValues)
class SiteRootDashboardOwnerModifiedValues(ContextRequestViewAdapter):
    """Site root dashboard modified contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()),
                        Or(Eq(catalog['role:owner'], self.request.principal.id),
                           Eq(catalog['role:contributor'], self.request.principal.id)))
            params = params | query if params else query
        yield from unique_iter(map(lambda x: IWorkflowVersions(x).get_last_versions()[0],
                                   CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                                limit=50,
                                                                                sort_index='modified_date',
                                                                                reverse=True))))


#
# All my contents menu
#

@viewlet_config(name='my-contents.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IContentManagementMenu, permission=VIEW_SYSTEM_PERMISSION, weight=5)
@viewletmanager_config(name='my-contents.menu', layer=IAdminLayer, provides=IMyDashboardMenu)
@implementer(IMyDashboardMenu)
class SiteRootMyDashboardMenu(MenuItem):
    """Site root 'my contents' dashboard menu"""

    label = _("My contents")
    icon_class = 'fa-user'
    url = '#'


#
# My favorites
# Dashboard of owned and modified contents which can be updated
#

@viewlet_config(name='my-favorites.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IMyDashboardMenu, permission=VIEW_SYSTEM_PERMISSION, weight=3)
class SiteRootFavoritesMenu(MenuItem):
    """Site root favorites dashboard menu"""

    label = _("My favorites")
    icon_class = None
    url = '#my-favorites.html'


@implementer(ISiteRootDashboardTable)
class SiteRootFavoritesTable(BaseDashboardTable):
    """Site root favorites table"""

    _single_title = _("CONTRIBUTOR - {0} favorite")
    _plural_title = _("CONTRIBUTOR - {0} favorites")


@adapter_config(required=(ISiteRoot, IPyAMSLayer, SiteRootFavoritesTable),
                provides=IValues)
class SiteRootFavoritesValues(ContextRequestViewAdapter):
    """Site root favorites values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        profile = IAdminProfile(self.request.principal)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['oid'], profile.favorites or ()))
            params = params | query if params else query
        yield from unique_iter(map(get_last_version,
                                   CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                                sort_index='modified_date',
                                                                                reverse=True))))


@pagelet_config(name='my-favorites.html',
                context=ISiteRoot, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootFavoritesView(AdminView, ContainerView):
    """Site root favorites view"""

    table_class = SiteRootFavoritesTable


@adapter_config(required=(ISiteRoot, IPyAMSLayer, SiteRootFavoritesView),
                provides=IPageHeader)
class SiteRootFavoritesHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root favorites header adapter"""

    icon_class = 'fa fa-fw fa-star'

    title = _("Your favorite contents")


#
# My preparations
# Dashboard of owned and modified contents which can be updated
#

@viewlet_config(name='my-preparations.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IMyDashboardMenu, permission=VIEW_SYSTEM_PERMISSION, weight=5)
class SiteRootPreparationsMenu(MenuItem):
    """Site root preparations dashboard menu"""

    label = _("My drafts")
    icon_class = None
    url = '#my-preparations.html'


@implementer(ISiteRootDashboardTable)
class SiteRootPreparationsTable(BaseDashboardTable):
    """Site root preparations table"""

    _single_title = _("CONTRIBUTOR - {0} content in preparation")
    _plural_title = _("CONTRIBUTOR - {0} contents in preparation")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootPreparationsTable), provides=IValues)
class SiteRootPreparationsValues(ContextRequestViewAdapter):
    """Site root preparations values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()) & 
                        Or(Eq(catalog['role:owner'], self.request.principal.id),
                           Eq(catalog['role:contributor'], self.request.principal.id)),
                        Eq(catalog['workflow_state'], workflow.initial_state))
            params = params | query if params else query
        yield from unique_iter(CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                            sort_index='modified_date',
                                                                            reverse=True)))


@pagelet_config(name='my-preparations.html', context=ISiteRoot, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootPreparationsView(AdminView, ContainerView):
    """Site root preparations view"""

    table_class = SiteRootPreparationsTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootPreparationsView), provides=IPageHeader)
class SiteRootPreparationsHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root preparations header adapter"""

    icon_class = 'fa fa-fw fa-user'

    title = _("Your prepared contents")


#
# My submissions
# Dashboard of owned and modified contents which are waiting manager action
#

@viewlet_config(name='my-submissions.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IMyDashboardMenu, permission=VIEW_SYSTEM_PERMISSION, weight=7)
class SiteRootSubmissionsMenu(MenuItem):
    """Site root submissions dashboard menu"""

    label = _("My submissions")
    icon_class = None
    url = '#my-submissions.html'


@implementer(ISiteRootDashboardTable)
class SiteRootSubmissionsTable(BaseDashboardTable):
    """Site root submissions table"""

    _single_title = _("CONTRIBUTOR - {0} submitted content")
    _plural_title = _("CONTRIBUTOR - {0} submitted contents")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootSubmissionsTable), provides=IValues)
class SiteRootSubmissionsValues(ContextRequestViewAdapter):
    """Site root submissions values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()) & 
                        Or(Eq(catalog['role:owner'], self.request.principal.id),
                           Eq(catalog['role:contributor'], self.request.principal.id)),
                        Any(catalog['workflow_state'], workflow.waiting_states))
            params = params | query if params else query
        yield from unique_iter(CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                            sort_index='modified_date',
                                                                            reverse=True)))


@pagelet_config(name='my-submissions.html', context=ISiteRoot, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootSubmissionsView(AdminView, ContainerView):
    """Site root submissions view"""

    table_class = SiteRootSubmissionsTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootSubmissionsView), provides=IPageHeader)
class SiteRootSubmissionsHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root submissions header adapter"""

    icon_class = 'fa fa-fw fa-user'

    title = _("Your submitted contents")


#
# My publications
# Dashboard of owned and modified contents which are published
#

@viewlet_config(name='my-publications.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IMyDashboardMenu, permission=VIEW_SYSTEM_PERMISSION, weight=10)
class SiteRootPublicationsMenu(MenuItem):
    """Site root publications dashboard menu"""

    label = _("My publications")
    icon_class = None
    url = '#my-publications.html'


@implementer(ISiteRootDashboardTable)
class SiteRootPublicationsTable(BaseDashboardTable):
    """Site root publications table"""

    _single_title = _("CONTRIBUTOR - {0} published content")
    _plural_title = _("CONTRIBUTOR - {0} published contents")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootPublicationsTable), provides=IValues)
class SiteRootPublicationsValues(ContextRequestViewAdapter):
    """Site root publications values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()) & 
                        Or(Eq(catalog['role:owner'], self.request.principal.id),
                           Eq(catalog['role:contributor'], self.request.principal.id)),
                        Any(catalog['workflow_state'], workflow.published_states))
            params = params | query if params else query
        yield from unique_iter(CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                            sort_index='modified_date',
                                                                            reverse=True)))


@pagelet_config(name='my-publications.html', context=ISiteRoot, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootPublicationsView(AdminView, ContainerView):
    """Site root publications view"""

    table_class = SiteRootPublicationsTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootPublicationsView), provides=IPageHeader)
class SiteRootPublicationsHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root publications header adapter"""

    icon_class = 'fa fa-fw fa-user'

    title = _("Your published contents")


#
# My retired contents
# Dashboard of owned and modified contents which are retired
#

@viewlet_config(name='my-retired-contents.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IMyDashboardMenu, permission=VIEW_SYSTEM_PERMISSION, weight=15)
class SiteRootRetiredMenu(MenuItem):
    """Site root retired contents dashboard menu"""

    label = _("My retired contents")
    icon_class = None
    url = '#my-retired-contents.html'


@implementer(ISiteRootDashboardTable)
class SiteRootRetiredContentsTable(BaseDashboardTable):
    """Site root retired contents table"""

    _single_title = _("CONTRIBUTOR - {0} retired content")
    _plural_title = _("CONTRIBUTOR - {0} retired contents")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootRetiredContentsTable), provides=IValues)
class SiteRootRetiredContentsValues(ContextRequestViewAdapter):
    """Site root retired contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()) & 
                        Or(Eq(catalog['role:owner'], self.request.principal.id),
                           Eq(catalog['role:contributor'], self.request.principal.id)),
                        Any(catalog['workflow_state'], workflow.retired_states))
            params = params | query if params else query
        yield from unique_iter(CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                            sort_index='modified_date',
                                                                            reverse=True)))


@pagelet_config(name='my-retired-contents.html', context=ISiteRoot, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootRetiredContentsView(AdminView, ContainerView):
    """Site root retired contents view"""

    table_class = SiteRootRetiredContentsTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootRetiredContentsView), provides=IPageHeader)
class SiteRootRetiredContentsHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root retired contents header adapter"""

    icon_class = 'fa fa-fw fa-user'

    title = _("Your retired contents")


#
# My archived contents
# Dashboard of owned and modified contents which are archived
#

@viewlet_config(name='my-archived-contents.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IMyDashboardMenu, permission=VIEW_SYSTEM_PERMISSION, weight=20)
class SiteRootArchivedMenu(MenuItem):
    """Site root archived contents dashboard menu"""

    label = _("My archived contents")
    icon_class = None
    url = '#my-archived-contents.html'


@implementer(ISiteRootDashboardTable)
class SiteRootArchivedContentsTable(BaseDashboardTable):
    """Site root archived contents table"""

    _single_title = _("CONTRIBUTOR - {0} archived content")
    _plural_title = _("CONTRIBUTOR - {0} archived contents")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootArchivedContentsTable), provides=IValues)
class SiteRootArchivedContentsValues(ContextRequestViewAdapter):
    """Site root archived contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        principal_id = self.request.principal.id
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()) & 
                        Or(Eq(catalog['role:owner'], principal_id),
                           Eq(catalog['role:contributor'], principal_id)),
                        Any(catalog['workflow_state'], workflow.archived_states))
            params = params | query if params else query
        yield from unique_iter(map(lambda x: sorted((version for version in
                                                     IWorkflowVersions(x).get_versions(IWorkflow(x).archived_states)
                                                     if principal_id in (version.owner | version.contributors)),
                                                    key=lambda x: IWorkflowState(x).version_id,
                                                    reverse=True)[0],
                                   CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                                sort_index='modified_date',
                                                                                reverse=True))))


@pagelet_config(name='my-archived-contents.html', context=ISiteRoot, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootArchivedContentsView(AdminView, ContainerView):
    """Site root archived contents view"""

    table_class = SiteRootArchivedContentsTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootArchivedContentsView), provides=IPageHeader)
class SiteRootArchivedContentsHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root archived contents header adapter"""

    icon_class = 'fa fa-fw fa-user'

    title = _("Your archived contents")


#
# All interventions
#

@viewlet_config(name='all-interventions.menu',
                context=ISiteRoot, layer=IAdminLayer,
                manager=IContentManagementMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
@viewletmanager_config(name='all-interventions.menu',
                       layer=IAdminLayer, provides=IAllContentsMenu)
@implementer(IAllContentsMenu)
class SiteRootAllContentsMenu(MenuItem):
    """Site root 'all contents' dashboard menu"""

    label = _("Other interventions")
    css_class = 'open'
    icon_class = 'fa-pencil-square'
    url = '#'


#
# Last publications
# Dashboard of all published contents
#

@viewlet_config(name='all-publications.menu',
                context=ISiteRoot, layer=IAdminLayer,
                manager=IAllContentsMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
class SiteRootAllPublicationsMenu(MenuItem):
    """Site root published contents dashboard menu"""

    label = _("Last publications")
    icon_class = None
    url = '#all-publications.html'


@implementer(ISiteRootDashboardTable)
class SiteRootAllPublicationsTable(BaseDashboardTable):
    """Site root published contents table"""

    _single_title = _("CONTRIBUTORS - {0} published content")

    @property
    def _plural_title(self):
        if len(self.values) <= 50:
            return _("CONTRIBUTORS - Last {0} published contents")
        else:
            return _("CONTRIBUTORS - Last published contents (in the limit of 50)")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootAllPublicationsTable), provides=IValues)
class SiteRootAllPublicationsValues(ContextRequestViewAdapter):
    """Site root published contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            workflow = IWorkflow(tool)
            query = And(Eq(catalog['parents'], intids.register(tool)),
                        Any(catalog['content_type'], CONTENT_TYPES.keys()) & 
                        Any(catalog['workflow_state'], workflow.published_states))
            params = params | query if params else query
        yield from unique_iter(CatalogResultSet(CatalogQuery(catalog).query(params,
                                                                            limit=50,
                                                                            sort_index='modified_date',
                                                                            reverse=True)))


@pagelet_config(name='all-publications.html',
                context=ISiteRoot, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootAllPublicationsView(AdminView, ContainerView):
    """Site root published contents view"""

    table_class = SiteRootAllPublicationsTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootAllPublicationsView),
                provides=IPageHeader)
class SiteRootAllPublicationsHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root published contents header adapter"""

    icon_class = 'fa fa-fw fa-pencil-square'

    title = _("Last published contents")


#
# Last updates
# Dashboard of all updated contents
#

@viewlet_config(name='all-updates.menu',
                context=ISiteRoot, layer=IAdminLayer,
                manager=IAllContentsMenu, weight=20,
                permission=VIEW_SYSTEM_PERMISSION)
class SiteRootAllUpdatesMenu(MenuItem):
    """Site root updated contents dashboard menu"""

    label = _("Last updates")
    icon_class = None
    url = '#all-updates.html'


@implementer(ISiteRootDashboardTable)
class SiteRootAllUpdatesTable(BaseDashboardTable):
    """Site root updated contents table"""

    _single_title = _("CONTRIBUTORS - {0} updated content")

    @property
    def _plural_title(self):
        if len(self.values) <= 50:
            return _("CONTRIBUTORS - Last {0} updated contents")
        else:
            return _("CONTRIBUTORS - Last updated contents (in the limit of 50)")


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootAllUpdatesTable),
                provides=IValues)
class SiteRootAllUpdatesValues(ContextRequestViewAdapter):
    """Site root updated contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = None
        for tool in get_all_utilities_registered_for(IBaseSharedTool):
            query = Eq(catalog['parents'], intids.register(tool)) & \
                    Any(catalog['content_type'], CONTENT_TYPES.keys())
            params = params | query if params else query
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        limit=50,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='all-updates.html',
                context=ISiteRoot, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SiteRootAllUpdatesView(AdminView, ContainerView):
    """Site root updated contents view"""

    table_class = SiteRootAllUpdatesTable


@adapter_config(context=(ISiteRoot, IPyAMSLayer, SiteRootAllUpdatesView),
                provides=IPageHeader)
class SiteRootAllUpdatesHeaderAdapter(DefaultPageHeaderAdapter):
    """Site root updated contents header adapter"""

    icon_class = 'fa fa-fw fa-pencil-square'

    title = _("Last updated contents")
