#
# -*- encoding: utf-8 -*-
#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration.interfaces import IIllustrationTarget
from pyams_content.component.theme import ICollectionsManagerTarget, IThemesManagerTarget
from pyams_content.component.theme.interfaces import ITagsManagerTarget
from pyams_content.features.footer.interfaces import IFooterTarget
from pyams_content.features.header.interfaces import IHeaderTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.redirect.interfaces import IRedirectionManagerTarget
from pyams_content.features.share.interfaces import ISocialShareManagerTarget
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION, OPERATOR_ROLE, WEBMASTER_ROLE
from pyams_content.root.interfaces import ISiteRoot, ISiteRootBackOfficeConfiguration, \
    ISiteRootConfiguration, ISiteRootRoles, ISiteRootToolsConfiguration
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_portal.interfaces import DESIGNER_ROLE, IPortalContext
from pyams_security.interfaces import IDefaultProtectionPolicy, IGrantedRoleEvent, \
    ISecurityManager, SYSTEM_ADMIN_ROLE
from pyams_security.property import RolePrincipalsFieldProperty
from pyams_security.security import ProtectedObject
from pyams_skin.configuration import BackOfficeConfiguration, Configuration
from pyams_skin.interfaces.configuration import IBackOfficeConfiguration, IConfiguration
from pyams_skin.skin import UserSkinnableContent
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.interfaces.site import ISiteRootFactory
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.site import BaseSiteRoot
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'

from pyams_content import _


@implementer(IDefaultProtectionPolicy, ISiteRoot, ISiteRootRoles, IPortalContext,
             ITagsManagerTarget, ICollectionsManagerTarget, IThemesManagerTarget,
             IIllustrationTarget, IHeaderTarget, IFooterTarget, ISocialShareManagerTarget,
             IRedirectionManagerTarget, IPreviewTarget)
class SiteRoot(ProtectedObject, BaseSiteRoot, UserSkinnableContent):
    """Main site root"""

    __roles__ = (SYSTEM_ADMIN_ROLE, WEBMASTER_ROLE, OPERATOR_ROLE, DESIGNER_ROLE)

    roles_interface = ISiteRootRoles

    managers = RolePrincipalsFieldProperty(ISiteRootRoles['managers'])
    webmasters = RolePrincipalsFieldProperty(ISiteRootRoles['webmasters'])
    operators = RolePrincipalsFieldProperty(ISiteRootRoles['operators'])
    designers = RolePrincipalsFieldProperty(ISiteRootRoles['designers'])

    content_name = _("Site root")


@utility_config(provides=ISiteRootFactory)
class SiteRootFactory(object):
    """Default site root factory"""

    def __new__(cls):
        return SiteRoot


@factory_config(provided=IConfiguration)
@implementer(ISiteRootConfiguration)
class SiteRootConfiguration(Configuration):
    """Site root configuration"""

    public_url = FieldProperty(ISiteRootConfiguration['public_url'])


@factory_config(provided=IBackOfficeConfiguration)
@implementer(ISiteRootBackOfficeConfiguration)
class SiteRootBackOfficeConfiguration(BackOfficeConfiguration):
    """Site root back-office configuration"""


@subscriber(IGrantedRoleEvent)
def handle_granted_role(event):
    """Add principals to operators group when a role is granted"""
    role_id = event.role_id
    if (role_id == OPERATOR_ROLE) or (not role_id.startswith('pyams.')):
        return
    root = get_parent(event.object, ISiteRoot)
    if not root.operators:
        return
    security = get_utility(ISecurityManager)
    for principal_id in root.operators:
        if not principal_id:
            continue
        group = security.get_principal(principal_id, info=False)
        if event.principal_id not in group.principals:
            group.principals = group.principals | {event.principal_id}


@adapter_config(context=ISiteRoot, provides=IFormContextPermissionChecker)
class SiteRootPermissionChecker(ContextAdapter):
    """Site root permission checker"""

    edit_permission = MANAGE_SITE_ROOT_PERMISSION


#
# Tools configuration
#

@factory_config(ISiteRootToolsConfiguration)
class SiteRootToolsConfiguration(Persistent):
    """Site root tools configuration"""

    tables_name = None

    pictograms_table_name = None

    tools_name = None

    views_tool_name = None
    forms_tool_name = None
    imagemaps_tool_name = None
    logos_tool_name = None

    news_tool_name = None
    topics_tool_name = None
    calendar_tool_name = None


SITEROOT_TOOLS_CONFIGURATION_KEY = 'pyams_config.tools.configuration'


@adapter_config(context=ISiteRoot, provides=ISiteRootToolsConfiguration)
def site_root_tools_configuration_factory(context):
    """Site root tools configuration factory"""
    return get_annotation_adapter(context, SITEROOT_TOOLS_CONFIGURATION_KEY, ISiteRootToolsConfiguration)
