#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import locale

from zope.interface import Interface

from pyams_content.interfaces import IBaseContent, MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference import IReferenceTable
from pyams_content.reference.interfaces import IReferenceInfo, IReferenceManager
from pyams_content.reference.zmi.table import ReferenceTableContentsTable
from pyams_form.form import AJAXEditForm
from pyams_i18n.interfaces import II18n
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_skin.interfaces.container import ITableElementName
from pyams_skin.interfaces.viewlet import IBreadcrumbItem, ITopLinksViewletManager
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.toplinks import TopLinksMenu, TopLinksViewlet
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.registry import get_local_registry
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='references.menu', layer=IAdminLayer, manager=ITopLinksViewletManager,
                permission=MANAGE_SITE_ROOT_PERMISSION, weight=70)
class ReferencesMenu(TopLinksViewlet):
    """References tables menu"""

    label = ''
    css_class = 'top-menu bordered margin-top-10'
    dropdown_label = _("References")

    def update(self):
        super(ReferencesMenu, self).update()
        registry = get_local_registry()
        references = registry.queryUtility(IReferenceManager)
        if references is not None:
            for table in references.values():
                menu = TopLinksMenu(self.context, self.request, self.__parent__, self)
                menu.label = II18n(table).query_attribute('short_name', request=self.request) or table.__name__
                menu.url = absolute_url(table, self.request, 'admin#contents.html')
                self.viewlets.append(menu)
            self.viewlets.sort(key=lambda x: locale.strxfrm(x.label))


@adapter_config(context=(IReferenceManager, IAdminLayer), provides=IBreadcrumbItem)
class ReferenceManagerBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """References tables manager breadcrumb adapter"""

    label = _("References tables")
    url = None


@adapter_config(context=(IReferenceInfo, Interface), provides=ITableElementName)
class ReferenceInfoTableElementName(ContextRequestAdapter):
    """Reference table element name"""

    @property
    def name(self):
        return II18n(self.context).query_attribute('title', request=self.request)


#
# Reference table item views
#

class ReferenceInfoAJAXEditForm(AJAXEditForm):
    """Base reference info edit form"""

    table_factory = ReferenceTableContentsTable

    def get_ajax_output(self, changes):
        output = super(ReferenceInfoAJAXEditForm, self).get_ajax_output(changes)
        if 'title' in changes.get(IBaseContent, ()):
            parent = get_parent(self.context, IReferenceTable)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(parent, self.request, self.table_factory, self.context))
        return output
