#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from zope.location.interfaces import IContained
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

# import packages
from pyams_i18n.schema import I18nTextLineField
from zope.annotation.interfaces import IAttributeAnnotatable
from zope.interface import Interface
from zope.schema import Datetime, TextLine

from pyams_content import _


#
# Custom permissions
#

MANAGE_SITE_ROOT_PERMISSION = 'pyams.ManageSiteRoot'
'''Permission required to manage main site root properties'''

MANAGE_SITE_PERMISSION = 'pyams.ManageSite'
'''Permission required to manager inner site or blog properties'''

MANAGE_TOOL_PERMISSION = 'pyams.ManageTool'
'''Permission required to manager shared tool properties'''

CREATE_CONTENT_PERMISSION = 'pyams.CreateContent'
'''Permission required to create a new content'''

CREATE_VERSION_PERMISSION = 'pyams.CreateVersion'
'''Permission required to create a new version of an existing content'''

MANAGE_CONTENT_PERMISSION = 'pyams.ManageContent'
'''Permission required to manager properties of an existing content'''

COMMENT_CONTENT_PERMISSION = 'pyams.CommentContent'
'''Permission required to add comments on an existing content'''

PUBLISH_CONTENT_PERMISSION = 'pyams.PublishContent'
'''Permission required to publish or retire an existing content'''

USE_PUBLIC_API_PERMISSION = 'pyams.UsePublicAPI'
'''Permission required to use public APIs'''


#
# Custom roles
#

WEBMASTER_ROLE = 'pyams.Webmaster'
'''Webmaster role has all permissions on all contents'''

PILOT_ROLE = 'pyams.Pilot'
'''Pilot role is allowed to manage tools configuration and permissions'''

MANAGER_ROLE = 'pyams.Manager'
'''Manager role is allowed to manage contents workflow'''

OWNER_ROLE = 'pyams.Owner'
'''Content owner role is allowed to manage content properties until publication'''

CONTRIBUTOR_ROLE = 'pyams.Contributor'
'''Contributor role is allowed to create new contents'''

READER_ROLE = 'pyams.Reader'
'''Reader role is allowed to read and comment contents while still in draft state'''

OPERATOR_ROLE = 'pyams.Operator'
'''Operator role is allowed to access management interface'''

GUEST_ROLE = 'pyams.Guest'
'''Guest role is allowed to view contents'''

API_ROLE = 'pyams.API'
'''API client role is allowed to use public APIs'''


#
# Sort orders
#

RELEVANCE_ORDER = 'relevance'
CREATION_DATE_ORDER = 'created_date'
UPDATE_DATE_ORDER = 'modified_date'
PUBLICATION_DATE_ORDER = 'publication_date'
FIRST_PUBLICATION_DATE_ORDER = 'first_publication_date'
CONTENT_PUBLICATION_DATE_ORDER = 'content_publication_date'
VISIBLE_PUBLICATION_DATE_ORDER = 'visible_publication_date'
EXPIRATION_DATE_ORDER = 'expiration_date'

VIEW_ORDERS = (
    {'id': CREATION_DATE_ORDER, 'title': _("Published version creation date")},
    {'id': UPDATE_DATE_ORDER, 'title': _("Published version last update date")},
    {'id': PUBLICATION_DATE_ORDER, 'title': _("Current version publication date")},
    {'id': FIRST_PUBLICATION_DATE_ORDER, 'title': _("Current version first publication date")},
    {'id': CONTENT_PUBLICATION_DATE_ORDER, 'title': _("Content first publication date")},
    {'id': VISIBLE_PUBLICATION_DATE_ORDER, 'title': _("Visible publication date")},
    {'id': EXPIRATION_DATE_ORDER, 'title': _("Expiration date")}
)

VIEW_ORDER_VOCABULARY = SimpleVocabulary([SimpleTerm(item['id'], title=item['title'])
                                          for item in VIEW_ORDERS])
USER_VIEW_ORDERS = (
    {'id': RELEVANCE_ORDER, 'title': _("Relevance (on user search; if not on visible "
                                       "publication date)")},
    {'id': CREATION_DATE_ORDER, 'title': _("Published version creation date")},
    {'id': UPDATE_DATE_ORDER, 'title': _("Published version last update date")},
    {'id': PUBLICATION_DATE_ORDER, 'title': _("Current version publication date")},
    {'id': FIRST_PUBLICATION_DATE_ORDER, 'title': _("Current version first publication date")},
    {'id': CONTENT_PUBLICATION_DATE_ORDER, 'title': _("Content first publication date")},
    {'id': VISIBLE_PUBLICATION_DATE_ORDER, 'title': _("Visible publication date")},
    {'id': EXPIRATION_DATE_ORDER, 'title': _("Expiration date")}
)

USER_VIEW_ORDER_VOCABULARY = SimpleVocabulary([
    SimpleTerm(item['id'], title=item['title'])
    for item in USER_VIEW_ORDERS
])


#
# Base content interfaces
#

class IBaseContent(IContained, IAttributeAnnotatable):
    """Base content interface"""

    __name__ = TextLine(title=_("Unique key"),
                        description=_("WARNING: this key can't be modified after creation!!! Spaces, "
                                      "uppercase letters ou accentuated characters will be replaced "
                                      "automatically."),
                        required=True)

    title = I18nTextLineField(title=_("Title"),
                              description=_("Visible label used to display content"),
                              required=True)

    short_name = I18nTextLineField(title=_("Short name"),
                                   description=_("Short name used in breadcrumbs"),
                                   required=True)


class IBaseContentInfo(Interface):
    """Base content info interface"""

    created_date = Datetime(title=_("Creation date"),
                            required=False,
                            readonly=True)

    modified_date = Datetime(title=_("Modification date"),
                             required=False,
                             readonly=False)
