#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import field

from pyams_content.features.share import ISocialShareItem, SocialShareItem
from pyams_content.features.share.interfaces import ISocialShareManager, ISocialShareManagerTarget
from pyams_content.features.share.zmi.container import SocialShareContainerTable, SocialShareContainerView
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.pictograms.zmi.widget import PictogramSelectFieldWidget
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm

from pyams_content import _


@viewlet_config(name='add-social-share.action', context=ISocialShareManagerTarget, layer=IPyAMSLayer,
                view=SocialShareContainerView, manager=IWidgetTitleViewletManager,
                permission=MANAGE_SITE_ROOT_PERMISSION, weight=1)
class SocialShareItemAddAction(ToolbarAction):
    """Social network share item add action"""

    label = _("Add network share")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-social-share.html'
    modal_target = True


@pagelet_config(name='add-social-share.html', context=ISocialShareManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-social-share.json', context=ISocialShareManagerTarget, layer=IPyAMSLayer, base=AJAXAddForm)
class SocialShareItemAddForm(AdminDialogAddForm):
    """Social network share item add form"""

    dialog_class = 'modal-large'
    legend = _("Add social network share")
    icon_css_class = 'fa fa-fw fa-share-alt'

    fields = field.Fields(ISocialShareItem).omit('__parent__', '__name__', 'active')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def create(self, data):
        return SocialShareItem()

    def add(self, object):
        ISocialShareManager(self.context).append(object)

    def nextURL(self):
        return absolute_url(self.context, self.request, 'social-share.html')


@pagelet_config(name='properties.html', context=ISocialShareItem, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='properties.json', context=ISocialShareItem, layer=IPyAMSLayer)
class SocialShareItemPropertiesEditForm(AdminDialogEditForm):
    """Social network share item properties edit form"""

    dialog_class = 'modal-large'
    prefix = 'social_share_properties.'

    legend = _("Edit social network share properties")
    icon_css_class = 'fa fa-fw fa-share-alt'

    fields = field.Fields(ISocialShareItem).omit('__parent__', '__name__', 'active')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def get_ajax_output(self, changes):
        output = super(SocialShareItemPropertiesEditForm, self).get_ajax_output(changes)
        updated = changes.get(ISocialShareItem, ())
        if updated:
            target = get_parent(self.context, ISocialShareManagerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(target, self.request, SocialShareContainerTable, self.context))
        return output
