#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget

from pyams_content.features.search.interfaces import ISearchManagerInfo
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.root import ISiteRoot
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuDivider, MenuItem
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

from pyams_content import _


@viewlet_config(name='search.divider', context=ISiteRoot, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_SITE_ROOT_PERMISSION, weight=699)
class SearchManagerMenuDivider(MenuDivider):
    """Search manager menu divider"""


@viewlet_config(name='search-manager.menu', context=ISiteRoot, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_SITE_ROOT_PERMISSION, weight=700)
class SearchManagerMenu(MenuItem):
    """Search manager menu"""

    label = _("Search settings...")
    icon_class = 'fa-search'
    url = 'search-manager.html'
    modal_target = True


@pagelet_config(name='search-manager.html', context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='search-manager.json', context=ISiteRoot, layer=IPyAMSLayer)
class SearchManagerPropertiesEditForm(AdminDialogEditForm):
    """Search manager properties edit form"""

    prefix = 'search_manager.'

    legend = _("Search engine settings")
    dialog_class = 'modal-large'

    fields = field.Fields(ISearchManagerInfo).omit('references')
    fields['enable_tags_search'].widgetFactory = SingleCheckBoxFieldWidget
    fields['enable_collections_search'].widgetFactory = SingleCheckBoxFieldWidget

    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def getContent(self):
        return ISearchManagerInfo(self.context)

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'tags_search', self.widgets,
                                         ('enable_tags_search', 'tags_search_target'),
                                         legend=_("Enable search by tag"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=ISearchManagerInfo['enable_tags_search']))
        self.add_group(NamedWidgetsGroup(self, 'collections_search', self.widgets,
                                         ('enable_collections_search',
                                          'collections_search_target'),
                                         legend=_("Enable search by collection"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=ISearchManagerInfo[
                                             'enable_collections_search']))
        super(SearchManagerPropertiesEditForm, self).updateGroups()