#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from hypatia.interfaces import ICatalog
from hypatia.query import Contains, Eq, NotAny, Or
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content import _
from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.search.interfaces import ISearchFolder, ISearchFolderRoles, \
    ISearchFormRequestParams
from pyams_content.interfaces import GUEST_ROLE, MANAGER_ROLE, MANAGE_SITE_PERMISSION
from pyams_content.shared.view import IViewQuery, ViewQuery, WfView
from pyams_content.shared.view.interfaces import IViewUserQuery
from pyams_content.shared.view.portlet import SEARCH_EXCLUDED_ITEMS
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import INegotiator
from pyams_portal.interfaces import DESIGNER_ROLE, IPortalContext
from pyams_sequence.interfaces import ISequentialIntIds
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, NullAdapter, adapter_config
from pyams_utils.registry import get_utility
from pyams_utils.request import check_request
from pyams_workflow.interfaces import IWorkflowVersions


@implementer(ISearchFolder, ISearchFolderRoles, IIllustrationTarget, ILinkIllustrationTarget,
             IPortalContext, IPreviewTarget)
class SearchFolder(WfView):
    """Search folder"""

    __roles__ = (MANAGER_ROLE, DESIGNER_ROLE, GUEST_ROLE)
    roles_interface = ISearchFolderRoles

    content_name = _("Search folder")

    handle_short_name = True
    handle_header = True
    handle_description = True

    sequence_name = ''  # use default sequence generator
    sequence_prefix = ''

    short_name = FieldProperty(ISearchFolder['short_name'])

    selected_content_types = FieldProperty(ISearchFolder['selected_content_types'])
    selected_datatypes = FieldProperty(ISearchFolder['selected_datatypes'])

    order_by = FieldProperty(ISearchFolder['order_by'])
    visible_in_list = FieldProperty(ISearchFolder['visible_in_list'])
    navigation_title = FieldProperty(ISearchFolder['navigation_title'])

    @staticmethod
    def is_deletable():
        return True

    def get_results(self, context, sort_index=None, reverse=None, limit=None,
                    start=0, length=None, ignore_cache=False, get_count=False, request=None,
                    aggregates=None, settings=None, **kwargs):
        if not ignore_cache:
            if request is None:
                request = check_request()
            ignore_cache = bool(request.params)
        return super().get_results(context, sort_index, reverse, limit, start,
                                   length, ignore_cache, get_count, request,
                                   aggregates, settings, **kwargs)


@adapter_config(required=ISearchFolder,
                provides=IWorkflowVersions)
class SearchFolderVersions(NullAdapter):
    """Disable versions on search folder"""


@adapter_config(context=ISearchFolder, provides=IFormContextPermissionChecker)
class SearchFolderPermissionChecker(ContextAdapter):
    """Search folder edit permission checker"""

    edit_permission = MANAGE_SITE_PERMISSION


@adapter_config(required=ISearchFolder,
                provides=IViewQuery)
class SearchFolderQuery(ViewQuery):
    """Search folder query adapter"""

    def get_params(self, context, request=None, **kwargs):
        return super().get_params(context, request, get_user_params=True, **kwargs)


@adapter_config(name='exclusions',
                required=ViewQuery,
                provides=IViewUserQuery)
class SearchFolderExclusionsQuery(ContextAdapter):
    """Search folder exclusions query

    This adapter is looking into request's annotations for items which should be excluded
    from search.
    """

    @staticmethod
    def get_user_params(request):
        # check for results excluded by previous views
        if request is not None:
            excluded_items = request.annotations.get(SEARCH_EXCLUDED_ITEMS)
            if excluded_items:
                catalog = get_utility(ICatalog)
                yield NotAny(catalog['oid'], excluded_items)


@adapter_config(name='user_search',
                required=ViewQuery,
                provides=IViewUserQuery)
class SearchFolderUserQuery(ContextAdapter):
    """Search folder user query

    This adapter is looking for any fulltext search entered by user
    """

    @staticmethod
    def get_user_params(request):
        params = request.params
        fulltext = params.get('user_search')
        if fulltext:
            catalog = get_utility(ICatalog)
            # check +oid in user query
            if fulltext.startswith('+'):
                sequence = get_utility(ISequentialIntIds)
                oid = sequence.get_full_oid(fulltext)
                yield Eq(catalog['oid'], oid)
            else:
                # check fulltext query
                negotiator = get_utility(INegotiator)
                query_params = []
                for lang in {request.registry.settings.get('pyramid.default_locale_name', 'en'),
                             request.locale_name,
                             negotiator.server_language} | negotiator.offered_languages:
                    index_name = 'title:{0}'.format(lang)
                    if index_name in catalog:
                        index = catalog[index_name]
                        if index.check_query(fulltext):
                            query_params.append(
                                Contains(index, ' and '.join((w + '*'
                                                              for w in fulltext.split()))))
                yield Or(*query_params)


@adapter_config(name='content_type',
                required=ViewQuery,
                provides=IViewUserQuery)
class SearchFolderContentTypeQuery(ContextAdapter):
    """Search folder content-type query"""

    @staticmethod
    def get_user_params(request):
        content_type = request.params.get('content_type')
        if content_type:
            catalog = get_utility(ICatalog)
            yield Eq(catalog['content_type'], content_type)


@adapter_config(name='content_type',
                required=(ISearchFolder, IPyAMSUserLayer),
                provides=ISearchFormRequestParams)
class SearchFormContentTypeRequestParams(ContextRequestAdapter):
    """Search form content-type request params"""

    def get_params(self):
        content_type = self.request.params.get('content_type')
        if content_type:
            yield {
                'name': 'content_type',
                'value': content_type
            }


@adapter_config(name='data_type',
                required=ViewQuery,
                provides=IViewUserQuery)
class SearchFolderDataTypeQuery(ContextAdapter):
    """Search folder data-type query"""

    @staticmethod
    def get_user_params(request):
        data_type = request.params.get('data_type')
        if data_type:
            catalog = get_utility(ICatalog)
            yield Eq(catalog['data_type'], data_type)


@adapter_config(name='data_type',
                required=(ISearchFolder, IPyAMSUserLayer),
                provides=ISearchFormRequestParams)
class SearchFormDataTypeRequestParams(ContextRequestAdapter):
    """Search form data-type request params"""

    def get_params(self):
        data_type = self.request.params.get('data_type')
        if data_type:
            yield {
                'name': 'data_type',
                'value': data_type
            }
