#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.view import view_config
from pyramid_chameleon.interfaces import IChameleonTranslate
from z3c.form import button, field
from zope.interface import Interface
from zope.schema import Bool, Text

from pyams_content.features.review import ReviewComment
from pyams_content.features.review.interfaces import IReviewComments, IReviewManager, \
    IReviewTarget
from pyams_content.interfaces import COMMENT_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common import IWfSharedContentRoles
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.schema import CloseButton
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.interfaces import ISecurityManager
from pyams_security.interfaces.profile import IPublicProfile
from pyams_security.schema import PrincipalsSet
from pyams_skin.interfaces.viewlet import IContextActions
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_template.template import get_view_template, template_config
from pyams_utils.date import format_datetime, get_age
from pyams_utils.registry import get_utility, query_utility
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.interfaces.menu import IContentManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import InnerAdminView


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Review request form
#

@viewlet_config(name='ask-review.menu',
                context=IReviewTarget, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, permission=MANAGE_CONTENT_PERMISSION, weight=10)
class WfSharedContentReviewMenu(ToolbarMenuItem):
    """Shared content review menu"""

    label = _("Ask for review...")
    label_css_class = 'fa fa-fw fa-eye'

    url = 'ask-review.html'
    modal_target = True


class ISharedContentReviewInfo(Interface):
    """Shared content review infos"""

    reviewers = PrincipalsSet(title=_("Sought principals"),
                              description=_("List of principals from which a review is "
                                            "requested"),
                              required=True)

    comment = Text(title=_("Comment"),
                   description=_("Comment associated with this request"),
                   required=True)

    notify_all = Bool(title=_("Notify all reviewers"),
                      description=_("If 'yes', selected reviewers will be notified by mail of "
                                    "your request, even if they were already members of the "
                                    "reviewers group. Otherwise, only new reviewers will be "
                                    "notified"),
                      default=True,
                      required=True)


class ISharedContentReviewButtons(Interface):
    """Shared content review form buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    review = button.Button(name='review', title=_("Ask for content review"))


@pagelet_config(name='ask-review.html', context=IReviewTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='ask-review.json',
             context=IReviewTarget, layer=IPyAMSLayer, base=AJAXAddForm)
class WfSharedContentReviewForm(AdminDialogAddForm):
    """Shared content review form"""

    legend = _("Content review request")
    icon_css_class = 'fa fa-fw fa-eye'

    fields = field.Fields(ISharedContentReviewInfo)
    buttons = button.Buttons(ISharedContentReviewButtons)

    edit_permission = MANAGE_CONTENT_PERMISSION

    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    def updateWidgets(self, prefix=None):
        super(WfSharedContentReviewForm, self).updateWidgets(prefix)
        if 'comment' in self.widgets:
            self.widgets['comment'].rows = 6

    def updateActions(self):
        super(WfSharedContentReviewForm, self).updateActions()
        if 'review' in self.actions:
            self.actions['review'].addClass('btn-primary')

    def createAndAdd(self, data):
        data = data.get(self, data)
        manager = IReviewManager(self.context, None)
        if manager is not None:
            return manager.ask_review(data.get('reviewers'),
                                      data.get('comment'),
                                      data.get('notify_all'))

    def get_ajax_output(self, changes):
        translate = self.request.localizer.translate
        if changes:
            return {
                'status': 'success',
                'message': translate(_("Request successful. {count} new notification(s) "
                                       "have been sent")).format(count=changes),
                'events': [{
                    'event': 'myams.refresh',
                    'options': {
                        'handler': 'PyAMS_content.review.updateComments'
                    }
                }]
            }
        else:
            return {
                'status': 'info',
                'message': translate(_("Request successful. No new notification have been sent")),
                'events': [{
                    'event': 'myams.refresh',
                    'options': {
                        'handler': 'PyAMS_content.review.updateComments'
                    }
                }]
            }


#
# Share contents comments
#

@viewlet_config(name='review-comments.menu', context=IReviewTarget, layer=IAdminLayer,
                manager=IContentManagementMenu, permission=COMMENT_CONTENT_PERMISSION, weight=30)
class SharedContentReviewCommentsMenu(MenuItem):
    """Shared content review comments menu"""

    label = _("Comments")
    icon_class = 'fa-comments-o'
    url = '#review-comments.html'

    badge_class = 'bg-color-info'

    def update(self):
        super(SharedContentReviewCommentsMenu, self).update()
        nb_comments = len(IReviewComments(self.context))
        self.badge = str(nb_comments)
        if nb_comments == 0:
            self.badge_class += ' hidden'


@pagelet_config(name='review-comments.html', context=IReviewTarget, layer=IPyAMSLayer,
                permission=COMMENT_CONTENT_PERMISSION)
@template_config(template='templates/review-comments.pt', layer=IPyAMSLayer)
class SharedContentReviewCommentsView(InnerAdminView):
    """Shared content review comments view"""

    legend = _("Review comments")

    comments = None
    security = None

    def update(self):
        super(SharedContentReviewCommentsView, self).update()
        self.comments = IReviewComments(self.context).values()
        self.security = get_utility(ISecurityManager)

    def get_principal(self, principal_id):
        return self.security.get_principal(principal_id)

    @staticmethod
    def get_avatar(principal):
        return IPublicProfile(principal).avatar

    @staticmethod
    def get_date(comment):
        return format_datetime(comment.creation_date)

    @staticmethod
    def get_age(comment):
        return get_age(comment.creation_date)


@view_config(name='get-last-review-comments.json',
             context=IReviewTarget, request_type=IPyAMSLayer,
             permission=COMMENT_CONTENT_PERMISSION, renderer='json', xhr=True)
@template_config(template='templates/review-comments-json.pt')
class ReviewCommentsView(SharedContentReviewCommentsView):
    """"Get review comments"""

    def __init__(self, request):
        self.request = request
        self.context = request.context

    template = get_view_template()

    def __call__(self):
        result = {'status': 'success',
                  'count': 0}
        comments = IReviewComments(self.context)
        previous_count = int(self.request.params.get('count', 0))
        current_count = len(comments)
        if previous_count == current_count:
            result['count'] = current_count
        else:
            self.comments = comments.values()[previous_count:]
            self.security = get_utility(ISecurityManager)
            comments_body = self.template(request=self.request,
                                          context=self.context,
                                          view=self,
                                          translate=query_utility(IChameleonTranslate))
            result.update({'content': comments_body,
                           'count': len(comments)})
        return result


@view_config(name='add-review-comment.json',
             context=IReviewTarget, request_type=IPyAMSLayer,
             permission=COMMENT_CONTENT_PERMISSION, renderer='json', xhr=True)
@template_config(template='templates/review-add-comment.pt')
class ReviewCommentAddForm(object):
    """Review comment add form"""

    def __init__(self, request):
        self.request = request
        self.context = request.context

    template = get_view_template()

    def __call__(self):
        request = self.request
        translate = request.localizer.translate
        comment_body = request.params.get('comment')
        if not comment_body:
            return {
                'status': 'error',
                'message': translate(_("Message is mandatory!"))
            }
        # add new comment
        comments = IReviewComments(request.context)
        comment = ReviewComment(owner=request.principal.id,
                                comment=request.params.get('comment'))
        roles = IWfSharedContentRoles(request.context, None)
        if roles is not None:
            comment.is_reviewer_comment = comment.owner in (roles.readers or ())
        comments.add_comment(comment)
        # return comment infos
        profile = IPublicProfile(request.principal)
        comment_body = self.template(request=request,
                                     context=self.context,
                                     translate=query_utility(IChameleonTranslate),
                                     options={'comment': comment,
                                              'profile': profile})
        return {
            'status': 'success',
            'callback': 'PyAMS_content.review.addCommentCallback',
            'options': {
                'content': comment_body,
                'count': len(comments)
            }
        }
