#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from zope.annotation.interfaces import IAttributeAnnotatable
from zope.container.interfaces import IContainer, IContained
from zope.interface.interfaces import IObjectEvent, ObjectEvent

# import packages
from pyams_security.schema import Principal, PrincipalsSet
from zope.container.constraints import contains, containers
from zope.interface import implementer, Interface, Attribute
from zope.schema import Text, TextLine, Choice, Datetime, Bool

from pyams_content import _


COMMENT_TYPES = {'request': _("Review request"),
                 'comment': _("Reviewer comment")}


class ICommentAddedEvent(IObjectEvent):
    """Comment added event interface"""

    comment = Attribute("New comment")


@implementer(ICommentAddedEvent)
class CommentAddedEvent(ObjectEvent):
    """Comment added event"""

    def __init__(self, object, comment):
        super(CommentAddedEvent, self).__init__(object)
        self.comment = comment


class IReviewComment(IContained, IAttributeAnnotatable):
    """Review comment interface"""

    containers('.IReviewComments')

    owner = Principal(title=_("Comment writer"),
                      required=True)

    reviewers = TextLine(title=_("Content reviewers"),
                         required=False)

    comment_type = Choice(title=_("Comment type"),
                          values=COMMENT_TYPES.keys(),
                          required=True,
                          default='comment')

    comment = Text(title=_("Comment body"),
                   required=True)

    is_reviewer_comment = Bool(title=_("Reviewer comment?"),
                               required=True,
                               default=False)

    creation_date = Datetime(title=_("Creation date"),
                             required=False)


REVIEW_COMMENTS_ANNOTATION_KEY = 'pyams_content.review_comments'


class IReviewComments(IContainer):
    """Review comments container interface"""

    contains(IReviewComment)

    reviewers = PrincipalsSet(title=_("Reviewers list"),
                              description=_("List of principals which reviewed the comment"),
                              required=False)

    def clear(self):
        """Remove all comments"""

    def add_comment(self, comment):
        """Add given comment to list"""


class IReviewManager(Interface):
    """Content review interface"""

    def ask_review(self, reviewers, comment, notify=True):
        """Ask for content review"""


class IReviewTarget(Interface):
    """Review target marker interface

    This interface is used to mark contents which can handle review.
    """
